package com.getmati.mati_sdk.ui.email.email_validation

import android.os.Bundle
import android.view.View
import android.widget.Toast
import androidx.annotation.ColorRes
import androidx.core.content.ContextCompat
import androidx.fragment.app.activityViewModels
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.FragmentEmailVerificationBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.email.EmailSharedVm
import com.getmati.mati_sdk.ui.email.EmailSharedVmFactory
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.phonevalidation.AttemptsExhaustedFragment
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.PassCodeView
import com.getmati.mati_sdk.widgets.hideKeyboard
import com.getmati.mati_sdk.widgets.symbolBorderColor
import com.getmati.mati_sdk.widgets.symbolTextColor
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel
import java.text.SimpleDateFormat
import java.util.*
import kotlin.coroutines.coroutineContext

internal class EmailVerificationFragment : KYCBaseFragment(R.layout.fragment_email_verification) {

    override val screenName get() = "EmailVerification"

    private val email by lazy { requireArguments().getString(ARG_EMAIL)!! }
    private val binding by viewBinding(FragmentEmailVerificationBinding::bind)

    private val emailVerificationVm by viewModel<EmailVerificationVm>()

    private val emailSharedVm: EmailSharedVm by activityViewModels {
        EmailSharedVmFactory(verificationActivity!!, requireArguments())
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setListeners()
        setValues()
        observeState()

    }

    private fun setListeners() {
        binding.passCode.onChangeListener = PassCodeView.OnChangeListener { code, isComplete ->
            binding.errorTv.visibility = View.INVISIBLE
            resetPassCodeColor()
            binding.passCode.setOnClickListener(null)
            if (isComplete) {
                binding.passCode.hideKeyboard()
                emailVerificationVm.verifyEmail(email, code)
            }
        }
    }

    private fun setValues() {
        binding.subtitleTv.text = String.format("%s %s", getString(R.string.subtitle_prefix_email_verification), email)
    }

    private fun observeState() {
        viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Main) {
            emailVerificationVm.state.collect {
                when (it) {
                    is EmailVerificationVm.State.Initial -> {
                        showInitialState()
                    }
                    is EmailVerificationVm.State.EmailVerificationInProgress -> {
                        onVerificationProgress()
                    }
                    is EmailVerificationVm.State.EmailVerificationError -> {
                        onVerificationError(it.error.type)
                    }
                    is EmailVerificationVm.State.EmailVerificationSuccess -> {
                        onVerificationSuccess()
                    }
                    is EmailVerificationVm.State.CodeResendInProgress -> {
                        onCodeResendProgress()
                    }
                    is EmailVerificationVm.State.CodeResendError -> {
                        onCodeResendError(it.error.type)
                    }
                    is EmailVerificationVm.State.CodeResendSuccess -> {
                        onCodeResendSuccess()
                    }
                }
            }
        }
    }

    private fun onVerificationProgress() {
        binding.passCode.isEnabled = false
        binding.passCode.isClickable = false
        resetPassCodeColor()
        binding.errorTv.visibility = View.INVISIBLE
        binding.progressBar.visibility = View.VISIBLE
        binding.resendTv.visibility = View.INVISIBLE
    }

    private fun onCodeResendError(error: MediaVerificationError) {
        emailVerificationVm.dropState()
        if (error == MediaVerificationError.OTHER) {
            Toast.makeText(requireContext(), getString(R.string.label_something_went_wrong), Toast.LENGTH_SHORT).show()
        } else {
            navigation.navigateTo(
                BaseErrorFragment.destination(
                    prepareMediaErrorScreenData(
                        title = getString(error.title),
                        subHeading = getString(error.subtitle),
                        primaryCTALabel = getString(R.string.label_try_again)
                    )
                )
            )
        }
    }

    private fun onCodeResendSuccess() {
        emailSharedVm.onCodeSent()
        emailVerificationVm.dropState()
    }

    private fun onVerificationError(error: MediaVerificationError) {
        when (error) {
            MediaVerificationError.EMAIL_WRONG_CONFIRMATION_CODE -> {
                emailSharedVm.increaseCodeAttemptsCount()
                showWrongCodeState()
            }
            MediaVerificationError.EMAIL_TOO_MANY_RESENDS -> {
                emailSharedVm.lockResendCount()
                showError(error)
            }
            MediaVerificationError.EMAIL_CONFIRMATION_CODE_TOO_MANY_ATTEMPTS -> {
                emailSharedVm.lockCodeAttemptsCount()
                if(emailSharedVm.resendLocked)
                    navigation.navigateTo(AttemptsExhaustedFragment.destination(AttemptsExhaustedFragment.Error.EMAIL_VERIFICATION_ATTEMPTS_EXHAUSTED))
                else
                    showTooManyAttemptsError()
            }
            else -> {
                emailVerificationVm.dropState()
                navigation.navigateTo(
                    BaseErrorFragment.destination(
                        prepareMediaErrorScreenData(
                            title = getString(error.title),
                            subHeading = getString(error.subtitle),
                            primaryCTALabel = getString(R.string.label_try_again)
                        )
                    )
                )
            }
        }
    }

    private fun onVerificationSuccess() {
        navigation.openNextStep()
    }

    private fun showInitialState() {
        if(emailSharedVm.codeAttemptLocked){
            showTooManyAttemptsError()
        }else {
            resetPassCodeView()
            binding.errorTv.visibility = View.INVISIBLE
            binding.progressBar.visibility = View.INVISIBLE
            showResend()
        }
    }

    private fun resetPassCodeView() {
        binding.passCode.reset()
        binding.passCode.setOnClickListener(null)
        binding.passCode.isEnabled = true
        binding.passCode.isClickable = true
        resetPassCodeColor()
    }

    private fun resetPassCodeColor() {
        changeCodeInputColor(R.color.matiViewElement)
    }

    private fun showResend() {
        if(emailSharedVm.resendLocked){
            binding.resendTv.visibility = View.INVISIBLE
        }else {
            binding.resendTv.visibility = View.VISIBLE
            if (emailSharedVm.resendIsAllowed()) {
                binding.resendTv.underline = true
                binding.resendTv.text = getString(R.string.label_resend)
                binding.resendTv.setSingleClickListener {
                    emailVerificationVm.resendCode(email)
                }
            } else {
                binding.resendTv.underline = false
                binding.resendTv.setOnClickListener(null)
                viewLifecycleOwner.lifecycleScope.launchWhenStarted {
                    trackCoolDown(
                            coolDownUntil = emailSharedVm.coolDownUntil,
                            onEachSecond = { time ->
                                binding.resendTv.text = String.format(
                                        "%s, %s",
                                        getString(R.string.label_sms_check_resend),
                                        time
                                )
                            },
                            onFinish = { if (binding.resendTv.visibility == View.VISIBLE) showResend() }
                    )
                }
            }
        }
    }

    private fun onCodeResendProgress() {
        binding.passCode.reset()
        binding.passCode.isEnabled = false
        binding.passCode.isClickable = false
        binding.passCode.hideKeyboard()
        resetPassCodeColor()
        binding.errorTv.visibility = View.INVISIBLE
        binding.progressBar.visibility = View.VISIBLE
        binding.resendTv.visibility = View.INVISIBLE
    }

    private fun showWrongCodeState() {
        if(emailSharedVm.codeAttemptLocked){
            if(emailSharedVm.resendLocked){
                navigation.navigateTo(AttemptsExhaustedFragment.destination(AttemptsExhaustedFragment.Error.EMAIL_VERIFICATION_ATTEMPTS_EXHAUSTED))
            }else {
                showTooManyAttemptsError()
            }
        }else {
            binding.passCode.isEnabled = true
            binding.passCode.isClickable = true
            binding.passCode.setOnClickListener { showInitialState() }
            showError(MediaVerificationError.EMAIL_WRONG_CONFIRMATION_CODE)
        }
    }

    private fun showTooManyAttemptsError() {
        binding.passCode.isEnabled = false
        binding.passCode.isClickable = false
        showError(MediaVerificationError.EMAIL_CONFIRMATION_CODE_TOO_MANY_ATTEMPTS)
    }

    private fun showError(error: MediaVerificationError){
        changeCodeInputColor(R.color.matiColorRed)
        binding.errorTv.visibility = View.VISIBLE
        binding.errorTv.text = getString(error.subtitle)
        binding.progressBar.visibility = View.INVISIBLE
        showResend()
    }

    private fun changeCodeInputColor(@ColorRes id: Int) {
        val color = ContextCompat.getColor(requireContext(), id)
        binding.passCode.symbolTextColor = color
        binding.passCode.symbolBorderColor = color
    }

    companion object {

        internal const val ARG_ATTEMPT_LIMIT = "ARG_ATTEMPT_LIMIT"
        internal const val ARG_COOL_DOWN = "ARG_COOL_DOWN"
        internal const val ARG_EMAIL = "ARG_EMAIL"

        fun destination(email: String, attemptLimit: Int, coolDown: Int): MatiDestination {
            return MatiDestination(R.id.to_email_verification, Bundle().apply {
                putString(ARG_EMAIL, email)
                putInt(ARG_ATTEMPT_LIMIT, attemptLimit)
                putInt(ARG_COOL_DOWN, coolDown)
            })
        }
    }
}


private const val COOL_DOWN_FORMAT = "mm:ss"
internal suspend fun trackCoolDown(coolDownUntil: Long, onEachSecond: (time: String) -> Unit, onFinish: () -> Unit) {
    var millisRemaining = coolDownUntil - System.currentTimeMillis()
    if (millisRemaining > 0) {
        val sdf = SimpleDateFormat(COOL_DOWN_FORMAT, Locale.getDefault())
        val date = Date()
        do {
            onEachSecond(sdf.format(date.apply { time = millisRemaining }))
            delay(1000)
            millisRemaining = coolDownUntil - System.currentTimeMillis()

        } while (millisRemaining > 0 && coroutineContext.isActive)
    }
    if (coroutineContext.isActive) onFinish()
}