package com.getmati.mati_sdk.ui.email.email_validation

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.ui.email.EmailVerificationRepository
import com.getmati.mati_sdk.server.RequestKit
import com.getmati.mati_sdk.server.verification.VerificationError
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch

internal class EmailVerificationVm(private val emailVerificationRepository: EmailVerificationRepository) : ViewModel() {

    private val _state = MutableStateFlow<State>(State.Initial)
    val state: StateFlow<State> get() = _state

    fun resendCode(email: String) {
        _state.value = State.EmailVerificationInProgress
        viewModelScope.launch(Dispatchers.IO) {
            _state.value = emailVerificationRepository.submitEmail(email).run {
                if (error == null) {
                    State.CodeResendSuccess
                } else {
                    State.CodeResendError(error)
                }
            }
        }
    }

    fun verifyEmail(email: String, code: String) {
        _state.value = State.EmailVerificationInProgress
        viewModelScope.launch(Dispatchers.IO) {
            _state.value = emailVerificationRepository.verifyEmail(email, code).run {
                if (error == null || error.type == MediaVerificationError.INPUT_LOCKED) {
                    State.EmailVerificationSuccess
                } else {
                    State.EmailVerificationError(error)
                }
            }
        }
    }

    fun dropState() {
        _state.value = State.Initial
    }

    sealed class State {
        object Initial : State()
        object EmailVerificationInProgress : State()
        object EmailVerificationSuccess : State()
        data class EmailVerificationError(val error: VerificationError) : State()
        object CodeResendInProgress : State()
        object CodeResendSuccess : State()
        data class CodeResendError(val error: VerificationError) : State()
    }
}