package com.getmati.mati_sdk.ui.esign

import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.viewModelScope
import com.getmati.mati_sdk.ui.ESignVerificationStep
import com.getmati.mati_sdk.ui.esign.model.State
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.launch


internal class ESignVM(private val eSignRepo: ESignRepo) : ViewModel() {

    private lateinit var verificationStep: ESignVerificationStep
    private val signedDocs = mutableMapOf<String, String>()
    private val docPathList = mutableListOf<String>()
    val screenStateFlow = MutableStateFlow<State>(State.None)
    val verificationStateFlow = MutableStateFlow<State>(State.None)
    val navigationFlow = MutableStateFlow(0)
    val stepCount by lazy {
        verificationStep.docsToSign.size
    }

    fun setVerificationStep(eSignVerificationStep: ESignVerificationStep) {
        verificationStep = eSignVerificationStep
    }

    fun startDownload(docIndex: Int) {
        val docToSign = verificationStep.docsToSign[docIndex]
        viewModelScope.launch(Dispatchers.IO) {
            screenStateFlow.value = State.Loading
            val path = eSignRepo.downloadDoc(docToSign)
            if (path == null) {
                screenStateFlow.value = State.ErrorDownloadPdf
            } else {
                docPathList += path
                screenStateFlow.value = State.Success(path)
            }
        }
    }

    fun isLastPage(docIndex: Int): Boolean {
        return verificationStep.docsToSign.lastIndex == docIndex
    }

    fun accept(name: String, docIndex: Int) {
        viewModelScope.launch(Dispatchers.IO) {
            val idTimeStamp = eSignRepo.next(verificationStep.docsToSign[docIndex])
            signedDocs += idTimeStamp
            verificationStateFlow.value = State.Loading
            val result = eSignRepo.accept(name, signedDocs)
            verificationStateFlow.value = if (result.error == null) {
                State.SignSuccess
            } else {
                State.Error(result.error)
            }
        }
    }

    fun next(docIndex: Int) {
        screenStateFlow.value = State.None
        val idTimeStamp = eSignRepo.next(verificationStep.docsToSign[docIndex])
        signedDocs += idTimeStamp
        navigationFlow.value = docIndex + 1
    }

    fun getDocumentPathByIndex(docIndex: Int): String? {
        return docPathList.getOrNull(docIndex)
    }

    override fun onCleared() {
        eSignRepo.clearPdfDir()
        super.onCleared()
    }
}

internal class ESignVMFactory(
    private val eSignRepo: ESignRepo
) : ViewModelProvider.Factory {
    override fun <T : ViewModel?> create(
        modelClass: Class<T>
    ): T {
        @Suppress("UNCHECKED_CAST")
        return ESignVM(eSignRepo) as T
    }
}


