package com.getmati.mati_sdk.ui.esign

import android.os.Bundle
import android.view.View
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.core.widget.doAfterTextChanged
import androidx.fragment.app.Fragment
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.FragmentESignBinding
import com.getmati.mati_sdk.ui.esign.model.State
import com.getmati.mati_sdk.ui.utils.Util
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch

internal class SignDocFragment : Fragment(R.layout.fragment_e_sign) {

    private val docIndex by lazy { requireArguments().getInt(ARG_INDEX, 0) }
    private val vm by lazy { (parentFragment as ESignHostFragment).eSignVM }

    private var isScrolledToBottom = false
    private val binding by viewBinding(FragmentESignBinding::bind)

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        vm.startDownload(docIndex)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setupUI()
        collectStates()
    }

    private fun setupUI() {
        var titleText = getString(R.string.label_signature_title)
        if (vm.stepCount > 1) {
            titleText += " (${docIndex + 1}/${vm.stepCount})"
        }
        binding.titleTextView.text = titleText
        binding.openPdfTextView.paint.isUnderlineText = true
        binding.openPdfTextView.setOnClickListener {
            val path = vm.getDocumentPathByIndex(docIndex)
            path?.let {
                Util.openPdf(requireContext(), it)
            }
        }

        if (vm.isLastPage(docIndex)) {
            binding.signatureContainer.isVisible = true
            binding.moveToNextContainer.isVisible = false
        } else {
            binding.moveToNextContainer.isVisible = true
            binding.signatureContainer.isVisible = false
        }
        binding.nameEditText.doAfterTextChanged {
            val input = it.toString()
            binding.signButton.isEnabled = input.isNotEmpty() && isScrolledToBottom
        }
        binding.nextButton.setOnClickListener {
            vm.next(docIndex)
        }
        binding.signButton.setOnClickListener {
            vm.accept(binding.nameEditText.text.toString(), docIndex)
        }
    }

    private fun collectStates() {
        viewLifecycleOwner.lifecycleScope.launch {
            vm.screenStateFlow.collect {
                when (it) {
                    is State.Loading -> {
                        binding.pdfLoadingProgressBar.isVisible = true
                    }
                    is State.Success -> {
                        binding.pdfLoadingProgressBar.isVisible = false
                        binding.pdfComponent.apply {
                            setFilePath(it.pdfPath)
                            onLastPageReached {
                                isScrolledToBottom = true
                                binding.nextButton.isEnabled = true
                                binding.signButton.isEnabled =
                                    binding.nameEditText.text.toString().isNotEmpty()
                            }
                        }
                    }
                    is State.ErrorDownloadPdf -> {
                        binding.pdfLoadingProgressBar.isVisible = false
                    }
                }
            }
        }

        viewLifecycleOwner.lifecycleScope.launch {
            vm.verificationStateFlow.collect {
                val loadingProgressBar = binding.loadingProgressBar
                when (it) {
                    is State.Loading -> loadingProgressBar.isVisible = true
                    is State.SignSuccess, is State.Error -> loadingProgressBar.isVisible = false
                }
            }
        }
    }

    companion object {
        private const val ARG_INDEX = "ARG_INDEX"

        @JvmStatic
        fun newInstance(docIndex: Int): SignDocFragment {
            return SignDocFragment().apply {
                arguments = bundleOf(ARG_INDEX to docIndex)
            }
        }
    }
}