package com.getmati.mati_sdk.ui.iprestrictions

import android.os.Bundle
import android.view.View
import androidx.activity.OnBackPressedCallback
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.FragmentVpnDetectedBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.server.verification.ErrorDetails
import com.getmati.mati_sdk.ui.common.ExitFragment
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel

internal class VpnDetectedFragment: KYCBaseFragment(R.layout.fragment_vpn_detected) {
    override val screenName by lazy { "vpnDetectedError" }
    private val binding by viewBinding(FragmentVpnDetectedBinding::bind)
    private val isCritical by lazy { requireArguments().getBoolean(ARG_CRITICAL) }
    private val vpnDetectedVM by viewModel<VpnDetectedVM>()

    override fun configureToolbar(toolbar: MatiToolbar) {
        super.configureToolbar(toolbar)
        toolbar.setBackImageVisibile(false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        binding.actionClose.run {
            visibility = if(isCritical) View.VISIBLE else View.GONE
            setOnClickListener{ navigation.navigateTo(ExitFragment.destination()) }
            requireActivity().onBackPressedDispatcher.addCallback(viewLifecycleOwner, object : OnBackPressedCallback(true) {
                override fun handleOnBackPressed() {
                    navigation.navigateTo(ExitFragment.destination())
                }
            })
        }

        binding.actionSkip.run {
            visibility = if(!isCritical) View.VISIBLE else View.GONE
            setOnClickListener{ navigation.openVerificationStep(0) }
        }

        binding.actionTryAgain.setOnClickListener {
            vpnDetectedVM.tryAgain()
        }

        viewLifecycleOwner.lifecycleScope.launch {
            vpnDetectedVM.stateFlow.collect {
                when(it) {
                    is VpnDetectedVM.State.Loaded -> when (it.error?.type) {
                        MediaVerificationError.CONNECTION_DATA_VPM_DETECTED -> {
                            (it.error.details as ErrorDetails.RestrictionErrorDetails).let { errorDetails ->
                                binding.actionClose.run {
                                    isEnabled = true
                                    visibility = if (isCritical) View.VISIBLE else View.GONE
                                }
                                binding.actionSkip.run {
                                    isEnabled = true
                                    visibility = if (!isCritical) View.VISIBLE else View.GONE
                                }
                                binding.actionTryAgain.visibility = if (errorDetails.triesCount > 0) View.VISIBLE else View.GONE
                                binding.loadingProgress.visibility = View.GONE

                            }
                        }
                        else -> navigation.back()
                    }
                    is VpnDetectedVM.State.Loading -> {
                        binding.actionSkip.isEnabled = false
                        binding.actionClose.isEnabled = false
                        binding.actionTryAgain.visibility = View.GONE
                        binding.loadingProgress.visibility = View.VISIBLE
                    }
                }
            }
        }
    }

    companion object {
        const val ARG_CRITICAL = "ARG_CRITICAL"

        fun destination(isCritical: Boolean): MatiDestination {
            return MatiDestination(R.id.to_vpn_detected, Bundle().apply {
                putBoolean(ARG_CRITICAL, isCritical)
            })
        }
    }
}

