package com.getmati.mati_sdk.ui.liveness

import android.os.Bundle
import android.view.View
import androidx.camera.core.CameraSelector
import androidx.camera.video.VideoRecordEvent
import androidx.core.net.toFile
import androidx.core.net.toUri
import androidx.core.view.isVisible
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Clicked
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.blink
import com.getmati.mati_sdk.databinding.FragmentLivenessBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.ESignVerificationStep
import com.getmati.mati_sdk.ui.camera.VideoCameraFragment
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.permission_denial.PermissionDenialInfoFragment
import com.getmati.mati_sdk.ui.utils.loadImageResource
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch

internal class LivenessFragment : VideoCameraFragment(R.layout.fragment_liveness) {

    override val screenName = "livenessCamera"

    private val binding by viewBinding(FragmentLivenessBinding::bind)
    override var lensFacing = CameraSelector.LENS_FACING_FRONT
    override val audioEnabled get() = false
    override val maxDuration get() = VIDEO_MAX_DURATION
    override val previewView get() = binding.previewView

    override fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.changeTheme(MatiToolbar.Theme.NONE)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        binding.livenessHintRvv.playLooping(
            "android.resource://${requireContext().packageName}/${R.raw.ic_liveness}".toUri()
        )
        viewLifecycleOwner.lifecycleScope.launch {
            state.collect {
                when(it){
                    is State.Initial -> { resetViews(false) }
                    is State.Preview -> resetViews(true)
                    is State.Recording -> startLiveness()
                    is State.Recorded -> onVideoTaken(it.event)
                    is State.CameraIsAbsent -> onCamerasNotFound()
                    is State.MaxDurationReached -> stopLiveness()
                }
            }
        }
        val isESign = verificationFlow.verificationSteps.any { it is ESignVerificationStep }
        binding.esignTxt.isVisible = isESign
        binding.make2CirclesWithNoseTxt.isVisible = !isESign

        binding.actionRotateCamera.setOnClickListener {
            flipCamera()
        }
    }

    private fun onVideoTaken(event: VideoRecordEvent.Finalize) {
        if (event.error == VideoRecordEvent.Finalize.ERROR_NONE
            || event.error == VideoRecordEvent.Finalize.ERROR_FILE_SIZE_LIMIT_REACHED
        ) {
            navigation.navigateTo(
                LivenessPreviewFragment.destination(
                    event.outputResults.outputUri.toFile().absolutePath,
                    lensFacing == CameraSelector.LENS_FACING_FRONT
                )
            )
        } else if (event.error == VideoRecordEvent.Finalize.ERROR_SOURCE_INACTIVE) {
            //ignore
        } else {
            viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Main) {
                navigation.navigateTo(
                    BaseErrorFragment.destination(
                        prepareMediaErrorScreenData(
                            title = getString(R.string.mati_hardware_error_heading),
                            subHeading = "",
                            primaryCTALabel = getString(R.string.label_try_again)
                        )
                    )

                )
            }
        }
        dropState()
    }

    private fun onCamerasNotFound() {
        navigation.navigateTo(
            BaseErrorFragment.destination(
                prepareMediaErrorScreenData(
                    title = getString(R.string.mati_hardware_error_heading),
                    subHeading = getString(R.string.mati_hardware_error_desc),
                    primaryCTALabel = getString(R.string.label_try_again)
                )
            )
        )
    }

    private fun resetActionIv(enable: Boolean) {
        binding.actionPrimary.run {
            binding.actionPrimary.isEnabled = enable
            visibility = View.VISIBLE
            loadImageResource(R.drawable.ic_start_liveness)
            if (enable) {
                setSingleClickListener {
                    track(UserAction("startRecordingButton", Clicked(), screenName))
                    handleFullDiskSpaceOrElse {
                        startRec()
                    }
                }
            } else {
                setOnClickListener(null)
            }
        }
    }

    private fun startLiveness() {
        binding.actionPrimary.run {
            isEnabled = false
            loadImageResource(R.drawable.ic_stop_liveness)
            setSingleClickListener {
                track(UserAction("stopRecordingButton", Clicked(), screenName))
                stopLiveness()
            }
            viewLifecycleOwner.lifecycleScope.launch {
                delay(VIDEO_MIN_DURATION)
                isEnabled = true
            }
        }
        showIndicator()
        binding.livenessHintTv.setText(R.string.label_stop)
    }

    private fun stopLiveness() {
        binding.actionPrimary.isEnabled = false
        stopRec()
    }

    private fun resetViews(enableAction: Boolean) {
        resetIndicator()
        binding.livenessHintTv.setText(R.string.label_start)
        resetActionIv(enableAction)
    }

    private fun showIndicator() {
        binding.recordingIndicatorImg.run {
            visibility = View.VISIBLE
            blink(500L)
        }
    }

    private fun resetIndicator() {
        binding.recordingIndicatorImg.clearAnimation()
        binding.recordingIndicatorImg.visibility = View.INVISIBLE
    }

    override fun onPermissionDenied(vararg permission: String) {
        onPermissionRationaleShown()
    }

    override fun onPermissionPermanentlyDenied(permission: String) {
        navigation.navigateTo(PermissionDenialInfoFragment.destination(permission))
    }

    companion object {
        private const val VIDEO_MIN_DURATION = 1_000L
        private const val VIDEO_MAX_DURATION = 15_000L

        fun destination() = MatiDestination(R.id.to_liveness, Bundle())
    }
}