package com.getmati.mati_sdk.ui.liveness

import android.graphics.SurfaceTexture
import android.media.MediaPlayer
import android.os.Build
import android.os.Bundle
import android.view.Surface
import android.view.TextureView
import android.view.View
import androidx.core.net.toUri
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.BiometryUploadEvent
import com.getmati.mati_sdk.analytics.events.Failed
import com.getmati.mati_sdk.analytics.events.Processed
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.centerCropMirrored
import com.getmati.mati_sdk.databinding.FragmentVideoUploadBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.ui.BiometryType
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.selfie.BiometricRuntimeRepo
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.android.ext.android.get
import org.koin.android.ext.android.inject
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import org.koin.core.qualifier.named

internal class VideoUploadFragment : KYCBaseFragment(R.layout.fragment_video_upload) {

    override val screenName get() = "VideoUpload"
    private val binding by viewBinding(FragmentVideoUploadBinding::bind)

    private val path by lazy { requireArguments().getString(ARG_VIDEO_PATH)!! }
    private val isFrontLens by lazy { requireArguments().getBoolean(ARG_FRONT_LENS) }
    private val frameTime by lazy { requireArguments().getInt(ARG_VIDEO_FRAME_TIME) }
    private val biometryType by lazy { BiometryType.valueOf(requireArguments().getString(ARG_BIOMETRY_TYPE)!!) }

    private val videoUploadVm by viewModel<VideoUploadVm> {
        val repo = when (biometryType) {
            BiometryType.SELFIE_VIDEO -> get<VideoRepo>(named<LivenessRepo>())
            else -> get<VideoRepo>(named<VoiceLivenessRepo>())
        }
        val biometricRepo by inject<BiometricRuntimeRepo>()
        parametersOf(repo, path, biometricRepo)
    }

    override fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.changeTheme(MatiToolbar.Theme.NONE)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        binding.textureView.surfaceTextureListener = object :TextureView.SurfaceTextureListener{
            override fun onSurfaceTextureAvailable(surface: SurfaceTexture, width: Int, height: Int) {
                val s = Surface(surface)
                MediaPlayer().apply {
                    setSurface(s)
                    setDataSource(requireContext(), path.toUri())
                    setOnPreparedListener {
                        binding.textureView.centerCropMirrored(it, isFrontLens)
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
                            it.seekTo(frameTime.toLong(), MediaPlayer.SEEK_CLOSEST)
                        else
                            it.seekTo(frameTime)
                    }
                    prepare()
                }
            }
            override fun onSurfaceTextureSizeChanged(surface: SurfaceTexture, width: Int, height: Int) {}
            override fun onSurfaceTextureDestroyed(surface: SurfaceTexture) = true
            override fun onSurfaceTextureUpdated(surface: SurfaceTexture) {}
        }
        viewLifecycleOwner.lifecycleScope.launch {
            videoUploadVm.state.collect {
                when (it) {
                    is VideoUploadVm.State.Error -> {
                        track(BiometryUploadEvent(Failed(it.error.type.id, null, 0), biometryType))
                        navigation.back()
                        navigation.navigateTo(
                            BaseErrorFragment.destination(
                                prepareMediaErrorScreenData(
                                    icon = it.error.type.iconId,
                                    title = getString(it.error.type.title),
                                    subHeading = getString(it.error.type.subtitle),
                                    primaryCTALabel = getString(R.string.label_try_again)
                                )
                            )
                        )
                    }
                    is VideoUploadVm.State.Success -> {
                        track(BiometryUploadEvent(Processed(), biometryType))
                        navigation.openNextStep()
                    }
                }
            }
        }
    }

    companion object {
        private const val ARG_VIDEO_PATH = "ARG_VIDEO_PATH"
        private const val ARG_FRONT_LENS = "ARG_FRONT_LENS"
        private const val ARG_VIDEO_FRAME_TIME = "ARG_VIDEO_FRAME_TIME"
        private const val ARG_BIOMETRY_TYPE = "ARG_BIOMETRY_TYPE"

        fun destination(path: String, frameTime: Int, isFrontLens: Boolean, type:BiometryType) = MatiDestination(
                R.id.to_video_upload_fragment,
                Bundle().apply {
                    putString(ARG_VIDEO_PATH, path)
                    putBoolean(ARG_FRONT_LENS, isFrontLens)
                    putInt(ARG_VIDEO_FRAME_TIME, frameTime)
                    putString(ARG_BIOMETRY_TYPE, type.name)
                }
        )
    }
}