package com.getmati.mati_sdk.ui.liveness

import android.os.Bundle
import android.view.View
import androidx.camera.core.CameraSelector
import androidx.camera.video.VideoRecordEvent
import androidx.core.net.toFile
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Clicked
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentVoiceLivenessBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.BiometryType
import com.getmati.mati_sdk.ui.camera.VideoCameraFragment
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.permission_denial.PermissionDenialInfoFragment
import com.getmati.mati_sdk.ui.selfie.BiometricRuntimeRepo
import com.getmati.mati_sdk.ui.utils.toFlagEmoji
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject

internal class VoiceLivenessFragment : VideoCameraFragment(R.layout.fragment_voice_liveness) {

    override val screenName get() = "voicelivenessCamera"
    private val binding by viewBinding(FragmentVoiceLivenessBinding::bind)

    private val biometricRuntimeRepo by inject<BiometricRuntimeRepo>()


    override var lensFacing = CameraSelector.LENS_FACING_FRONT
    override val audioEnabled get() = true
    override val maxDuration get() = VIDEO_MAX_DURATION
    override val previewView get() = binding.previewView

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        viewLifecycleOwner.lifecycleScope.launch {
            state.collect {
                when(it){
                    is State.Initial -> setValues(false)
                    is State.Preview -> setValues(true)
                    is State.Recording -> changeActionBehaviour()
                    is State.Recorded -> onVideoTaken(it.event)
                    is State.CameraIsAbsent -> onCamerasNotFound()
                    is State.MaxDurationReached -> stopVoiceLiveness()
                }
            }
        }
    }

    private fun onVideoTaken(event: VideoRecordEvent.Finalize) {
        if (event.error == VideoRecordEvent.Finalize.ERROR_NONE
            || event.error == VideoRecordEvent.Finalize.ERROR_FILE_SIZE_LIMIT_REACHED
        ) {
            navigation.navigateTo(
                VideoUploadFragment.destination(
                    event.outputResults.outputUri.toFile().absolutePath,
                    0,
                    true,
                    BiometryType.VOICE_LIVENESS
                )
            )
            binding.doneTv.visibility = View.VISIBLE
        } else if(event.error == VideoRecordEvent.Finalize.ERROR_SOURCE_INACTIVE){
            //ignore
        } else {
            navigation.navigateTo(
                BaseErrorFragment.destination(
                    prepareMediaErrorScreenData(
                        title = getString(R.string.mati_hardware_error_heading),
                        subHeading = "",
                        primaryCTALabel = getString(R.string.label_try_again)
                    )
                )
            )
        }
        dropState()
    }

    private fun onCamerasNotFound() {
        navigation.navigateTo(
            BaseErrorFragment.destination(
                prepareMediaErrorScreenData(
                    title = getString(R.string.mati_hardware_error_heading),
                    subHeading = getString(R.string.mati_hardware_error_desc),
                    primaryCTALabel = getString(R.string.label_try_again)
                )
            )
        )
        dropState()
    }

    override fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.changeTheme(MatiToolbar.Theme.NONE)
    }

    private fun setValues(enableAction: Boolean) {
        resetActionButton(enableAction)
        binding.digitsTv.text = biometricRuntimeRepo.voiceDataText
        var newSentense = requireContext().getString(R.string.LivenessVoiceCapturesayDigits)
        while (newSentense.contains("{", ignoreCase = true)) {
            val newFlag = newSentense.substringAfter("{").substringBefore('}')
            newSentense = newSentense.replaceFirst("\\{(.*?)\\}".toRegex(), newFlag.toFlagEmoji())
        }
        binding.titleTv.text = newSentense
    }

    private fun resetActionButton(enable: Boolean) {
        binding.doneTv.setText(R.string.label_start)
        if(enable) {
            binding.doneTv.setSingleClickListener {
                track(UserAction("startRecordingButton", Clicked(), screenName))
                handleFullDiskSpaceOrElse {
                    startRec()
                }
            }
        }else{
            binding.doneTv.setOnClickListener(null)
        }
    }

    private fun changeActionBehaviour() {
        binding.doneTv.isEnabled = false
        binding.doneTv.setText(R.string.label_done)
        binding.doneTv.setSingleClickListener {
            track(UserAction("doneButton", Clicked(), screenName))
            stopVoiceLiveness()
        }
        viewLifecycleOwner.lifecycleScope.launch {
            delay(VIDEO_MIN_DURATION)
            binding.doneTv.isEnabled = true
        }
    }

    private fun stopVoiceLiveness() {
        binding.doneTv.visibility = View.INVISIBLE
        stopRec()
    }

    override fun onPermissionDenied(vararg permission: String) {
        onPermissionRationaleShown()
    }

    override fun onPermissionPermanentlyDenied(permission: String) {
        navigation.navigateTo(PermissionDenialInfoFragment.destination(permission))
    }

    companion object {

        private const val VIDEO_MIN_DURATION = 1_000L
        private const val VIDEO_MAX_DURATION = 15_000L

        fun destination() = MatiDestination(R.id.to_voice_liveness, Bundle())
    }
}