package com.getmati.mati_sdk.ui.phonevalidation

import android.annotation.SuppressLint
import android.os.Bundle
import android.view.KeyEvent
import android.view.View
import android.view.inputmethod.InputMethodManager
import android.widget.TextView
import androidx.core.content.ContextCompat
import androidx.core.content.getSystemService
import androidx.core.widget.doAfterTextChanged
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.FragmentPhoneInputBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.document.CountriesRepo
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.phonevalidation.vm.PhoneInputVM
import com.getmati.mati_sdk.ui.utils.toFlagEmoji
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import org.koin.android.ext.android.inject
import org.koin.androidx.viewmodel.ext.android.viewModel

internal class PhoneInputFragment : KYCBaseFragment(R.layout.fragment_phone_input) {
    override val screenName = "phoneInput"

    private val isOptional by lazy { requireArguments().getBoolean(ARG_OPTIONAL) }
    private val binding by viewBinding(FragmentPhoneInputBinding::bind)
    private val phoneInputVM by lazy { requireActivity().viewModel<PhoneInputVM>().value }
    private val countriesRepo by inject<CountriesRepo>()

    @SuppressLint("SetTextI18n")
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setListeners()
        phoneInputVM.liveState.observe(viewLifecycleOwner) { state ->
            when (state) {
                is PhoneInputVM.State.SkipError ->{
                    phoneInputVM.dropState()
                    with(state.error.type){
                        navigation.navigateTo(
                            BaseErrorFragment.destination(
                                prepareMediaErrorScreenData(
                                    title = getString(title),
                                    subHeading = getString(subtitle),
                                    primaryCTALabel = getString(R.string.label_try_again)
                                )
                            )
                        )
                    }
                }
                is PhoneInputVM.State.SkipSuccess ->{
                    navigation.openNextStep()
                }
                is PhoneInputVM.State.PhoneInput -> {
                    showSkipIfNeeded()
                    binding.actionPrimary.run {
                        visibility = View.VISIBLE
                        isEnabled = true
                        setOnClickListener {
                            phoneInputVM.verifyPhone(lifecycleScope)
                        }
                    }

                    binding.phoneEditContainer.setBackgroundResource(R.drawable.bg_new_secondary_button)
                    binding.phoneEdit.setTextColor(ContextCompat.getColor(requireContext(), R.color.matiSecondaryText))
                    binding.phoneCodeTxt.setTextColor(ContextCompat.getColor(requireContext(), R.color.matiSecondaryText))

                    binding.errorTxt.visibility = View.INVISIBLE
                    binding.primaryProgress.visibility = View.INVISIBLE
                    binding.errorTxt.visibility = View.INVISIBLE

                    binding.phoneCodeTxt.text = "+${state.selectedCountry.dialingCode} "
                    binding.phoneEdit.run {
                        setText(state.phoneDigits)
                        setSelection(state.phoneDigits.length)
                    }

                    binding.countryCodeTxt.run {
                        text = "${state.selectedCountry.code.toFlagEmoji()} ${state.selectedCountry.code}"
                    }
                }
                is PhoneInputVM.State.Resend -> {
                    showSkipIfNeeded()
                    binding.actionPrimary.visibility = View.INVISIBLE
                    binding.errorTxt.visibility = View.INVISIBLE
                    binding.primaryProgress.visibility = View.INVISIBLE
                    binding.errorTxt.visibility = View.INVISIBLE

                    if (state.timeRemain != "") {
                        binding.actionSecondary.visibility = View.INVISIBLE
                        binding.resendTimerTxt.run {
                            visibility = View.VISIBLE
                            text = "${requireContext().getString(R.string.label_sms_check_resend)} ${state.timeRemain}"
                        }
                    } else {
                        binding.actionSecondary.run {
                            visibility = View.VISIBLE
                            setOnClickListener {
                                phoneInputVM.verifyPhone(lifecycleScope)
                            }
                        }
                        binding.resendTimerTxt.visibility = View.INVISIBLE
                    }

                    binding.phoneEditContainer.setBackgroundResource(R.drawable.bg_new_secondary_button)
                    binding.phoneEdit.setTextColor(ContextCompat.getColor(requireContext(), R.color.matiSecondaryText))
                    binding.phoneCodeTxt.setTextColor(ContextCompat.getColor(requireContext(), R.color.matiSecondaryText))

                    binding.phoneCodeTxt.text = "+${state.selectedCountry.dialingCode} "
                    binding.phoneEdit.run {
                        setText(state.phoneDigits)
                        setSelection(state.phoneDigits.length)
                    }

                    binding.countryCodeTxt.run {
                        text = "${state.selectedCountry.code.toFlagEmoji()} ${state.selectedCountry.code}"
                    }
                }
                is PhoneInputVM.State.Loading -> {
                    binding.primaryProgress.visibility = View.VISIBLE
                    binding.actionPrimary.visibility = View.INVISIBLE
                    binding.actionSecondary.visibility = View.INVISIBLE
                    binding.errorTxt.visibility = View.INVISIBLE
                    binding.resendTimerTxt.visibility = View.INVISIBLE
                    binding.skipTv.visibility = View.INVISIBLE
                }
                is PhoneInputVM.State.Error -> {
                    showSkipIfNeeded()
                    binding.primaryProgress.visibility = View.INVISIBLE
                    binding.actionPrimary.visibility = View.VISIBLE
                    binding.phoneEditContainer.setBackgroundResource(R.drawable.bg_red_stroke_rounded_corners)
                    binding.phoneEdit.setTextColor(ContextCompat.getColor(requireContext(), R.color.matiColorRed))
                    binding.phoneCodeTxt.setTextColor(ContextCompat.getColor(requireContext(), R.color.matiColorRed))
                    binding.actionPrimary.isEnabled = false

                    binding.phoneEdit.setOnFocusChangeListener { phoneEdit, isFocused ->
                        if (isFocused) {
                            phoneInputVM.setPhoneDigits((phoneEdit as TextView).text.toString())
                            requireContext().getSystemService<InputMethodManager>()?.showSoftInput(phoneEdit, 0)
                        }
                    }

                    binding.errorTxt.run {
                        setText(state.errorMessage)
                        visibility = View.VISIBLE
                    }

                }
                is PhoneInputVM.State.PhoneVerified -> {
                    // This line is required to reset state to `PhoneInput`
                    showSkipIfNeeded()
                    phoneInputVM.setCountryCode(state.selectedCountry)
                    navigation.navigateTo(SmsInputFragment.destination(state.phone, state.selectedCountry))
                }
                is PhoneInputVM.State.AttemptsExhausted -> {
                    navigation.navigateTo(AttemptsExhaustedFragment.destination(AttemptsExhaustedFragment.Error.PHONE_VERIFICATION_ATTEMPTS_EXHAUSTED))
                }
                PhoneInputVM.State.None -> showSkipIfNeeded()
            }
        }

        binding.countryCodeTxt.setOnClickListener {
            navigation.navigateTo(SelectPhoneCodeFragment.destination())
        }

        binding.phoneEdit.run {
            setOnEditorActionListener { _, actionId, _ ->
                if (actionId in listOf(KeyEvent.KEYCODE_ENTER, KeyEvent.KEYCODE_ENDCALL)) {
                    binding.focusCatcher.run {
                        requestFocus()
                        requireContext().getSystemService<InputMethodManager>()
                            ?.hideSoftInputFromWindow(this.windowToken, 0)
                    }
                }
                false
            }

            doAfterTextChanged {
                phoneInputVM.setPhoneDigits(it.toString())
            }
        }

        if (phoneInputVM.liveState.value == null || phoneInputVM.liveState.value == PhoneInputVM.State.None) {
            val country = verificationVm.initialInputs
                .filter { it.inputData?.countryCode != null }
                .mapNotNull { countriesRepo.getCountryByCode(it.inputData?.countryCode) }
                .firstOrNull()
            val code = country ?: countriesRepo.defaultFirstCountry
            code?.let { phoneInputVM.setCountryCode(it) }
        }
    }

    private fun showSkipIfNeeded(){
        binding.skipTv.visibility = if (isOptional) View.VISIBLE else View.GONE
    }

    private fun setListeners() {
        binding.skipTv.setSingleClickListener {
            phoneInputVM.skip()
        }
    }

    companion object {
        private const val ARG_OPTIONAL = "ARG_OPTIONAL"
        private const val ARG_ATTEMPTS_DELAY_MS = "ARG_ATTEMPTS_DELAY_MS"

        fun destination(optional: Boolean, attemptsDelayMs: Int): MatiDestination {
            return MatiDestination(R.id.to_phoneInputFragment, Bundle().apply {
                putBoolean(ARG_OPTIONAL, optional)
                putInt(ARG_ATTEMPTS_DELAY_MS, attemptsDelayMs)
            })
        }
    }
}