package com.getmati.mati_sdk.ui.phonevalidation

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.view.inputmethod.InputMethodManager
import android.widget.SearchView
import android.widget.TextView
import androidx.core.content.getSystemService
import androidx.lifecycle.*
import androidx.recyclerview.widget.RecyclerView
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Selected
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentSelectDialingCodeBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.Country
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.document.CountriesRepo
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject

internal class SelectPhoneCodeFragment : KYCBaseFragment(R.layout.fragment_select_dialing_code) {

    private val binding by viewBinding(FragmentSelectDialingCodeBinding::bind)
    private val phoneVerificationRepo by inject<PhoneVerificationRepo>()
    private val countriesRepo by inject<CountriesRepo>()

    override val screenName = "selectDialingCode"

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        val countryCode = countriesRepo.getCountryFromInput(verificationVm.initialInputs)?.code
        val allowedRegions = verificationFlow.ipValidation?.allowedRegions ?: emptyList()
        val countries = countriesRepo.reorderWithAllowedRegions(currentCountryCode = countryCode,
            allowedRegions = allowedRegions
        )
        val selectedLanguage = countryCode ?: countries.first().code

        val adapter = SelectPhoneCodeAdapter(viewLifecycleOwner, countries, selectedLanguage)
        binding.recyclerView.adapter = adapter
        adapter.selectedPhoneCode.observe(viewLifecycleOwner){
            phoneVerificationRepo.setCountryCode(it)
            navigation.back()
        }
        binding.search.let {
            it.requestFocus()
            viewLifecycleOwner.lifecycleScope.launch {
                delay(50)
                requireContext().getSystemService<InputMethodManager>()?.toggleSoftInput(InputMethodManager.SHOW_IMPLICIT, 0)
            }
            it.setOnQueryTextListener(adapter)
        }
    }

    companion object {
        fun destination(): MatiDestination {
            return MatiDestination(R.id.to_selectPhoneCode, Bundle())
        }
    }
}

internal class SelectPhoneCodeAdapter(val lifecycleOwner: LifecycleOwner, val allCountries: List<Country>, var selectedLanguageId: String) :
    RecyclerView.Adapter<PhoneCodeViewHolder>(), SearchView.OnQueryTextListener {
    val selectedPhoneCode = MutableLiveData<Country>()
    val query = MutableLiveData<String?>()

    val phones = MediatorLiveData<List<Country>>().apply {
        addSource(query){
            this.postValue(filterPhones(allCountries, it))
        }
    }

    init {
        phones.observe(lifecycleOwner){
            notifyDataSetChanged()
        }
        phones.postValue(allCountries)
    }

    override fun onCreateViewHolder(viewGroup: ViewGroup, viewType: Int): PhoneCodeViewHolder {
        return PhoneCodeViewHolder(LayoutInflater.from(viewGroup.context).inflate(R.layout.item_phone_code, viewGroup, false))
    }

    override fun onBindViewHolder(holder: PhoneCodeViewHolder, position: Int) {
        val savedPhones = phones.value

        val phone = if(savedPhones != null) {
            savedPhones[position]
        } else {
            null
        }

        holder.bind(phone!!, phone.code == selectedLanguageId) {
            selectedLanguageId = phone.code
            selectedPhoneCode.postValue(it)
        }
    }

    override fun onQueryTextSubmit(query: String?): Boolean {
        return true
    }

    override fun onQueryTextChange(newText: String?): Boolean {
        query.postValue(newText ?: "")
        return true
    }

    private fun filterPhones(phones: List<Country>, query: String?): List<Country> {
        return if(query.isNullOrBlank()) {
            phones
        } else {
            phones.filter { it.code.toLowerCase().startsWith(query.toLowerCase())
                    || it.name.toLowerCase().startsWith(query.toLowerCase()) }
        }
    }

    override fun getItemCount(): Int {
        return phones.value?.size ?: 0
    }
}

internal class PhoneCodeViewHolder(itemView: View) : RecyclerView.ViewHolder(itemView) {
    fun bind(phoneCode: Country, isSelected: Boolean, listener: (phoneCode: Country) -> Unit) {
        itemView.findViewById<TextView>(R.id.phoneCodeTxt).text = "${phoneCode.name} (+${phoneCode.dialingCode})"
        itemView.isSelected = isSelected
        itemView.setOnClickListener {
            track(UserAction("phoneCodeSelector", Selected(phoneCode.code), "selectDialingCode"))
            listener(phoneCode)
        }
    }
}