package com.getmati.mati_sdk.ui.phonevalidation

import android.annotation.SuppressLint
import android.os.Bundle
import android.telephony.PhoneNumberUtils
import android.view.View
import android.widget.TextView
import androidx.fragment.app.activityViewModels
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.FragmentSmsInputBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.Country
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.phonevalidation.vm.PhoneInputVM
import com.getmati.mati_sdk.ui.phonevalidation.vm.SmsCodeInputVM
import com.getmati.mati_sdk.ui.utils.findView
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.PassCodeView
import com.getmati.mati_sdk.widgets.hideKeyboard
import org.koin.androidx.viewmodel.ext.android.viewModel

internal class SmsInputFragment : KYCBaseFragment(R.layout.fragment_sms_input) {

    override val screenName = "phoneInput"

    private val binding by viewBinding(FragmentSmsInputBinding::bind)
    private val phoneDigits by lazy { requireArguments().getString(ARG_PHONE_DIGITS, "") }
    private val selectedCountry: Country by lazy { requireArguments().getParcelable(ARG_COUNTRY)!! }

    private val smsCodeInputVM by viewModel<SmsCodeInputVM>()
    private val phoneInputVM by lazy { requireActivity().viewModel<PhoneInputVM>().value } //TODO move counter logic to phoneInput repo, instead of making viewmodel shared @Arman Chatikyan

    @SuppressLint("SetTextI18n")
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        if (savedInstanceState == null) {
            smsCodeInputVM.resetAttemptsCount()
        }

        smsCodeInputVM.liveState.observe(viewLifecycleOwner) { state ->
            when (state) {
                null, is SmsCodeInputVM.State.None -> {
                    binding.progressBar.visibility = View.INVISIBLE
                    resetPassCodeView()
                }
                SmsCodeInputVM.State.Loading -> {
                    binding.passCode.isEnabled = false
                    binding.passCode.isEnabled = false
                    binding.passCode.isClickable = false
                    binding.passCode.changeColor(R.color.matiViewElement)
                    binding.progressBar.visibility = View.VISIBLE
                }
                is SmsCodeInputVM.State.Error -> {
                    binding.progressBar.visibility = View.INVISIBLE

                    binding.passCode.changeColor(R.color.matiColorRed)
                    binding.passCode.isEnabled = true
                    binding.passCode.isClickable = true
                    binding.passCode.setOnClickListener { resetPassCodeView() }

                    binding.errorTxt.visibility = View.VISIBLE


                    if (state.timeBeforeNextTry != "") {
                        binding.actionPrimary.visibility = View.INVISIBLE
                        binding.resendTimerTxt.visibility = View.VISIBLE
                        binding.resendTimerTxt.text =
                            "${getString(R.string.label_sms_check_resend)} ${state.timeBeforeNextTry}"
                    } else {
                        binding.actionPrimary.visibility = View.VISIBLE
                        binding.resendTimerTxt.visibility = View.INVISIBLE
                    }


                    binding.errorTxt.run {
                        visibility = View.VISIBLE
                        text = getString(state.errorId)
                    }
                }
                is SmsCodeInputVM.State.SmsVerified -> {
                    navigation.openNextStep()
                }
                is SmsCodeInputVM.State.TryAnotherPhoneNumber -> {
                    binding.progressBar.visibility = View.INVISIBLE

                    binding.passCode.changeColor(R.color.matiColorRed)
                    binding.passCode.isEnabled = false
                    binding.passCode.isClickable = false
                    binding.actionPrimary.visibility = View.VISIBLE
                    binding.resendTimerTxt.visibility = View.INVISIBLE
                    binding.errorTxt.run {
                        visibility = View.VISIBLE
                        setText(R.string.error_attempts_to_enter_code_have_been_exhausted)
                    }

                }
                is SmsCodeInputVM.State.AttemptsExhausted -> {
                    navigation.navigateTo(AttemptsExhaustedFragment.destination(AttemptsExhaustedFragment.Error.PHONE_VERIFICATION_ATTEMPTS_EXHAUSTED))
                }
            }
            findView<TextView>(R.id.subtitleTxt).text =
                "${getString(R.string.label_sms_check_sent)} ${
                    formatPhoneDigits(
                        selectedCountry,
                        phoneDigits
                    )
                }"

            phoneInputVM.liveCounter.observe(viewLifecycleOwner) {
                if (it != "") {
                    binding.actionPrimary.visibility = View.INVISIBLE
                    binding.resendTimerTxt.visibility = View.VISIBLE
                    binding.resendTimerTxt.text = "${getString(R.string.label_sms_check_resend)} $it"
                } else {
                    binding.actionPrimary.visibility = View.VISIBLE
                    binding.resendTimerTxt.visibility = View.INVISIBLE
                }
            }
        }

        binding.passCode.onChangeListener = PassCodeView.OnChangeListener { code, isComplete ->
            binding.errorTxt.visibility = View.INVISIBLE
            if (isComplete) {
                binding.passCode.hideKeyboard()
                smsCodeInputVM.verify(code, phoneDigits, selectedCountry)
            }
        }

        binding.actionPrimary.setOnClickListener {
            if (phoneInputVM.phoneInputCount < 3) {
                navigation.back()
            } else {
                navigation.navigateTo(AttemptsExhaustedFragment.destination(AttemptsExhaustedFragment.Error.PHONE_VERIFICATION_ATTEMPTS_EXHAUSTED))
            }
        }
    }


    private fun resetPassCodeView() {
        binding.passCode.reset()
        binding.passCode.setOnClickListener(null)
        binding.passCode.isEnabled = true
        binding.passCode.isClickable = true
        binding.passCode.changeColor(R.color.matiViewElement)
    }

    private fun formatPhoneDigits(selectedCountry: Country, phoneDigits: String): String {
        return try {
            PhoneNumberUtils.formatNumber(
                "+${selectedCountry.dialingCode}${phoneDigits.filter { it.isDigit() }}",
                selectedCountry.code
            )
        } catch (_: Exception) {
            "+${selectedCountry.dialingCode}${phoneDigits.filter { it.isDigit() }}"
        }
    }

    companion object {
        private const val ARG_PHONE_DIGITS = "ARG_PHONE_DIGITS"
        private const val ARG_COUNTRY = "ARG_COUNTRY"

        fun destination(phoneDigits: String, selectedCountry: Country): MatiDestination {
            return MatiDestination(R.id.to_smsInputFragment, Bundle().apply {
                putString(ARG_PHONE_DIGITS, phoneDigits)
                putParcelable(ARG_COUNTRY, selectedCountry)
            })
        }
    }
}