package com.getmati.mati_sdk.ui.phonevalidation.vm

import android.telephony.PhoneNumberUtils.formatNumber
import androidx.lifecycle.*
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataHolder
import com.getmati.mati_sdk.models.clean.Country
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.models.api.response.ApiResponse
import com.getmati.mati_sdk.server.api.PhoneInputApi
import com.getmati.mati_sdk.server.verification.VerificationError
import com.getmati.mati_sdk.ui.phonevalidation.PhoneVerificationRepo
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.channels.ReceiveChannel
import kotlinx.coroutines.channels.consumeEach
import kotlinx.coroutines.channels.ticker
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import java.text.SimpleDateFormat
import java.util.*

internal class PhoneInputVM(
    private val phoneVerificationRepo: PhoneVerificationRepo,
    private val prefetchDataHolder: PrefetchDataHolder,
    private val phoneInputApi: PhoneInputApi
) : ViewModel() {

    private val ATTEMPT_DELAY_MS = 60_000

    private val mutableLiveState = MutableLiveData<State>().apply { // Add country repo dependency
        postValue(State.None)
    }

    private val mutableLiveCounter = MutableLiveData<String>()

    private val formatter = SimpleDateFormat("mm:ss")

    private var mutablePhoneInputCount: Int = 0

    private var mutableLastVerifiedPhone = ""

    val phoneInputCount: Int
        get() {
            return mutablePhoneInputCount
        }

    val liveCounter: LiveData<String>
        get() = mutableLiveCounter

    val liveState = MediatorLiveData<State>().apply {
        addSource(mutableLiveState) {
            postValue(mergeStateAndCounter(it, liveCounter.value ?: ""))
        }
        addSource(liveCounter) {
            postValue(mergeStateAndCounter(mutableLiveState.value ?: State.None, it))
        }
    }

    init {
        viewModelScope.launch {
            phoneVerificationRepo.countrySelectionFlow.collect {
                setCountryCode(it)
            }
        }
    }

    private fun mergeStateAndCounter(naturalState: State, counterValue: String): State {
        return when {
            phoneInputCount > 3 -> State.AttemptsExhausted
            naturalState is State.PhoneInput -> if (phoneInputCount < 1 ||
                "${naturalState.selectedCountry.dialingCode}${naturalState.phoneDigits.filter { it.isDigit() }}" != mutableLastVerifiedPhone
            ) {
                naturalState
            } else {
                State.Resend(
                    naturalState.selectedCountry,
                    naturalState.phoneDigits,
                    naturalState.isValid,
                    counterValue
                )
            }
            else -> naturalState
        }
    }

    fun setCountryCode(givenCountry: Country) {
        liveState.value.let {
            when (it) {
                is State.Resend -> {
                    updateIfNeeded(formatPhoneDigits(givenCountry, it.phoneDigits))
                }
                is State.PhoneInput -> {
                    updateIfNeeded(formatPhoneDigits(givenCountry, it.phoneDigits))
                }
                else -> {
                    updateIfNeeded(formatPhoneDigits(givenCountry, ""))
                }
            }
        }
    }

    fun setPhoneDigits(givenPhoneDigits: String) {
        val selectedCountry = (mutableLiveState.value as? State.PhoneInput)?.selectedCountry

        when (val state = liveState.value) {
            is State.Error -> {
                mutableLiveState.postValue(
                    formatPhoneDigits(
                        selectedCountry ?: state.selectedCountry,
                        givenPhoneDigits
                    )
                )
            }
            is State.PhoneInput -> {
                if (state.phoneDigits != givenPhoneDigits) {
                    mutableLiveState.postValue(
                        formatPhoneDigits(
                            state.selectedCountry,
                            givenPhoneDigits
                        )
                    )
                }
            }
            is State.Resend -> {
                if (state.phoneDigits != givenPhoneDigits) {
                    mutableLiveState.postValue(
                        formatPhoneDigits(
                            state.selectedCountry,
                            givenPhoneDigits
                        )
                    )
                }
            }
            else -> {
            }
        }
    }

    fun setError(selectedCountry: Country, phoneDigits: String, error: Int) {
        updateIfNeeded(State.Error(selectedCountry, phoneDigits, error))
    }

    fun verifyPhone(scope: CoroutineScope) {
        liveState.value?.run {
            when (this) {
                is State.PhoneInput -> if (isValid) {
                    sendSms(selectedCountry, phoneDigits)
                } else {
                    updateIfNeeded(
                        State.Error(
                            selectedCountry,
                            phoneDigits,
                            R.string.error_invalid_phone_number
                        )
                    )
                }
                is State.Resend -> if (isValid) {
                    sendSms(selectedCountry, phoneDigits)
                } else {
                    updateIfNeeded(
                        State.Error(
                            selectedCountry,
                            phoneDigits,
                            R.string.error_invalid_phone_number
                        )
                    )
                }
                else -> {
                    updateIfNeeded(State.None)
                }
            }
        }
    }

    private fun sendSms(selectedCountry: Country, phoneDigits: String) {
        viewModelScope.launch {
            updateIfNeeded(State.Loading)
            val phone = phoneDigits.filter { it.isDigit() }
            val prefetchedData = prefetchDataHolder.prefetchedData
            val verificationId = prefetchedData.verificationId
            val response = phoneInputApi.sendSms(verificationId, phone, selectedCountry)
            if (response is ApiResponse.Success) {
                updateIfNeeded(State.PhoneVerified(selectedCountry, phone))
            } else {
                updateIfNeeded(
                    State.Error(
                        selectedCountry,
                        phoneDigits,
                        R.string.error_invalid_phone_number
                    )
                )
            }
            mutablePhoneInputCount += 1
            resetCounter("${selectedCountry.dialingCode}$phone")
        }
    }

    private fun formatPhoneDigits(selectedCountry: Country, phoneDigits: String): State {
        return try {
            val formattedDigits = formatNumber(
                "+${selectedCountry.dialingCode}${phoneDigits.filter { it.isDigit() }}",
                selectedCountry.code
            )
                .removePrefix("+${selectedCountry.dialingCode}")
            State.PhoneInput(
                selectedCountry,
                formattedDigits,
                formattedDigits.any { !it.isDigit() })
        } catch (_: Exception) {
            State.PhoneInput(selectedCountry, phoneDigits, false)
        }
    }

    private fun updateIfNeeded(newState: State) {
        if (newState != mutableLiveState.value) {
            mutableLiveState.postValue(newState)
        }
    }

    var ticker: ReceiveChannel<Unit>? = null

    private fun resetCounter(phone: String) {
        val targetTimestamp = System.currentTimeMillis() + ATTEMPT_DELAY_MS
        mutableLastVerifiedPhone = phone
        viewModelScope.launch {
            ticker?.cancel()
            ticker = ticker(1000, 0)
            ticker?.consumeEach {
                val timeRemainMs = targetTimestamp - System.currentTimeMillis()
                when {
                    phoneInputCount > 3 -> {
                        mutableLiveState.postValue(State.AttemptsExhausted)
                        ticker?.cancel()
                    }
                    timeRemainMs > 0 -> {
                        mutableLiveCounter.postValue(formatter.format(Date(timeRemainMs)))
                    }
                    else -> {
                        ticker?.cancel()
                        mutableLiveCounter.postValue("")
                    }
                }
            }
        }
    }

    fun skip() {
        mutableLiveState.value = State.Loading
        viewModelScope.launch(Dispatchers.IO) {
            mutableLiveState.postValue(
                phoneVerificationRepo.skip().run {
                    if (error == null || error.type == MediaVerificationError.INPUT_SKIPPED) {
                        State.SkipSuccess
                    } else {
                        State.SkipError(error)
                    }

                }
            )
        }
    }

    fun dropState() {
        mutableLiveState.value = State.None
    }

    sealed class State {
        internal object None : State()
        internal object SkipSuccess : State()
        internal data class SkipError(val error: VerificationError) : State()
        internal data class PhoneInput(
            val selectedCountry: Country,
            val phoneDigits: String,
            val isValid: Boolean
        ) : State()

        internal data class Resend(
            val selectedCountry: Country,
            val phoneDigits: String,
            val isValid: Boolean,
            val timeRemain: String
        ) : State()

        internal object Loading : State()
        internal data class Error(
            val selectedCountry: Country,
            val phoneDigits: String,
            val errorMessage: Int
        ) : State()

        internal object AttemptsExhausted : State()
        internal data class PhoneVerified(val selectedCountry: Country, val phone: String) : State()
    }
}