package com.getmati.mati_sdk.ui.phonevalidation.vm

import androidx.annotation.StringRes
import androidx.lifecycle.*
import com.getmati.mati_sdk.models.clean.Country
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.server.RequestKit
import com.getmati.mati_sdk.ui.phonevalidation.PhoneVerificationRepo
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.text.SimpleDateFormat

internal class SmsCodeInputVM(private val phoneVerificationRepo: PhoneVerificationRepo) : ViewModel() {

    private val mutableLiveState = MutableLiveData<State>().apply {
        postValue(State.None)
    }

    private val mutableLiveCounter = MutableLiveData<String>()

    private var attemptsCount: Int = 0

    fun resetAttemptsCount() {
        mutableLiveState.postValue(State.None)
        attemptsCount = 0
    }

    val liveState = MediatorLiveData<State>().apply {
        addSource(mutableLiveState) {
            postValue(when {
                attemptsCount >= 5 -> State.TryAnotherPhoneNumber
                it is State.SmsInput -> it.apply {
                    timeBeforeNextTry = mutableLiveCounter.value ?: ""
                }
                it is State.Error -> it.apply {
                    timeBeforeNextTry = mutableLiveCounter.value ?: ""
                }
                else -> it
            })
        }
        addSource(mutableLiveCounter) {
            val currentState = mutableLiveState.value
            postValue(when {
                attemptsCount > 5 -> State.TryAnotherPhoneNumber
                currentState is State.SmsInput -> currentState.apply {
                    timeBeforeNextTry = it ?: ""
                }
                currentState is State.Error -> currentState.apply {
                    timeBeforeNextTry = it ?: ""
                }
                else -> currentState ?: State.None
            })
        }
    }

    private fun updateStateIfNeeded(newState: State) {
        if (newState != mutableLiveState.value) {
            mutableLiveState.postValue(newState)
        }
    }

    fun verify(code: String, phoneDigits: String, selectedCountry: Country) {
        if (code.length != 4) return
        updateStateIfNeeded(State.Loading)
        viewModelScope.launch(Dispatchers.IO) {
            phoneVerificationRepo.verifyPhoneNumber(phoneDigits, selectedCountry, code).run {
                updateStateIfNeeded(
                    when (error?.type) {
                        null, MediaVerificationError.INPUT_LOCKED -> State.SmsVerified
                        MediaVerificationError.PHONE_TOO_MANY_ATTEMPTS,
                        MediaVerificationError.PHONE_TOO_MANY_RESENDS,
                        MediaVerificationError.PHONE_NOT_ENOUGH_PARAMS,
                        MediaVerificationError.PHONE_INVALID_PHONE_NUMBER ->
                            State.TryAnotherPhoneNumber
                        else -> State.Error(error.type.id, error.type.title, "")
                    }
                )
            }
        }
    }

    internal sealed class State {
        internal object None : State()
        internal data class SmsInput(val smsCode: String, var timeBeforeNextTry: String) : State()
        internal object Loading : State()
        internal data class Error(
            val smsCode: String,
            @StringRes val errorId: Int,
            var timeBeforeNextTry: String
        ) : State()

        internal object SmsVerified : State()
        internal object TryAnotherPhoneNumber : State()
        internal object AttemptsExhausted : State()
    }
}