package com.getmati.mati_sdk.ui.selfie

import android.graphics.Bitmap
import android.hardware.camera2.CameraCharacteristics
import android.net.Uri
import android.os.Bundle
import android.view.View
import androidx.activity.result.contract.ActivityResultContracts
import androidx.camera.core.ImageCaptureException
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.Constants
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.*
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragTakeSelfieBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.ui.BiometryType
import com.getmati.mati_sdk.ui.camera.PhotoCameraFragment
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.permission_denial.PermissionDenialInfoFragment
import com.getmati.mati_sdk.ui.utils.*
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.File
import java.io.FileOutputStream

internal class SelfieCameraFragment : PhotoCameraFragment(R.layout.frag_take_selfie), PhotoCapture {

    override val screenName = "selfieCamera"
    private val binding by viewBinding(FragTakeSelfieBinding::bind)

    override var lensFacing = CameraCharacteristics.LENS_FACING_FRONT
    override val flipHorizontally get() = lensFacing == CameraCharacteristics.LENS_FACING_FRONT
    override val viewFinder get() = binding.previewView

    override var latestTmpUri: Uri? = null

    override val takePhotoResultLauncher = registerForActivityResult(ActivityResultContracts.TakePicture()) { isSuccess ->
        if (isSuccess) {
            latestTmpUri?.let { _ ->
                val rotationResult = rotateAndScaleDown(requireContext().getImagePath(Constants.TEMP_FILE_NAME))
                if(rotationResult is BitmapTransformation.Success) {
                    val dstPath = requireContext().getImagePath("selfie")
                    rotationResult.bitmap.compress(Bitmap.CompressFormat.JPEG, 100, FileOutputStream(dstPath))
                    onPictureReady(dstPath, MediaSource.NATIVE_CAMERA)
                } else {
                    MediaVerificationError.OTHER.run {
                        navigation.navigateTo(
                            BaseErrorFragment.destination(
                                prepareMediaErrorScreenData(
                                    title = getString(title),
                                    subHeading = getString(subtitle),
                                    primaryCTALabel = getString(R.string.label_try_again)
                                )
                            )
                        )
                    }
                }
            }
        }
    }

    override fun onImageSaved(file: File) {
        view?.let {
            viewLifecycleOwner.lifecycleScope.launch(Dispatchers.IO) {
                context?.let { context ->
                    val rotationResult = rotateAndScaleDown(file.absolutePath)
                    if (rotationResult is BitmapTransformation.Success) {
                        val dstPath = context.getImagePath("selfie")
                        rotationResult.bitmap.compress(
                            Bitmap.CompressFormat.JPEG,
                            100,
                            FileOutputStream(dstPath)
                        )
                        withContext(Dispatchers.Main) {
                            binding.actionCapture.isEnabled = true
                            onPictureReady(dstPath, MediaSource.NATIVE_CAMERA)
                        }
                    } else {
                        withContext(Dispatchers.Main) {
                            MediaVerificationError.OTHER.run {
                                navigation.navigateTo(
                                    BaseErrorFragment.destination(
                                        prepareMediaErrorScreenData(
                                            title = getString(title),
                                            subHeading = getString(subtitle),
                                            primaryCTALabel = getString(R.string.label_try_again)
                                        )
                                    )
                                )
                            }
                        }
                    }
                }
            }
        }
    }

    override fun onImageCaptureException(exc: ImageCaptureException) {
        view?.let {
            viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Main) {
                binding.actionCapture.isEnabled = true
            }
            takePhotoWithOtherApp()
        }
    }

    override fun onUseCaseBindingFailed(exc: Exception) {
        takePhotoWithOtherApp()
    }

    override fun onPermissionDenied(vararg permission:String) {
        onPermissionRationaleShown()
    }

    override fun onPermissionPermanentlyDenied(permission:String) {
        navigation.navigateTo(PermissionDenialInfoFragment.destination(permission))
    }

    override fun onFrontBackCamerasNotFound() {
        navigation.navigateTo(
            BaseErrorFragment.destination(
                prepareMediaErrorScreenData(
                    title = getString(R.string.mati_hardware_error_heading),
                    subHeading = getString(R.string.mati_hardware_error_desc),
                    primaryCTALabel = getString(R.string.label_try_again)
                )
            )
        )
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        binding.actionCapture.setOnClickListener {
            it.isEnabled = false
            track(UserAction("capturePhotoButton", Clicked(), screenName))
            verificationActivity?.let {
                takePhoto()
            }
        }

        binding.actionRotateCamera.setOnClickListener {
            flipCamera()
        }
    }

    override fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.changeTheme(MatiToolbar.Theme.NONE)
    }

    private fun onPictureReady(pPath: String, source: MediaSource) {
        track(BiometryUploadEvent(MediaObtained(source), BiometryType.SELFIE))
        navigation.navigateTo(SelfiePreviewFragment.destination(pPath))
    }

    companion object {
        fun destination(): MatiDestination {
            return MatiDestination(R.id.to_selfieCamera, Bundle())
        }
    }
}