package com.getmati.mati_sdk.ui.selfie

import android.content.Context
import android.net.Uri
import android.os.Bundle
import android.view.View
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.BiometryUploadEvent
import com.getmati.mati_sdk.analytics.events.Failed
import com.getmati.mati_sdk.analytics.events.Processed
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentSelfieUploadBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.ui.BiometryType
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.ErrorScreenConstants
import com.getmati.mati_sdk.ui.error.prepareBaseErrorScreenData
import com.getmati.mati_sdk.ui.error.prepareMediaProcessingErrorScreenData
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import java.io.File
import java.net.HttpURLConnection

internal class SelfieUploadFragment : KYCBaseFragment(R.layout.fragment_selfie_upload) {

    override val screenName = "selfieUpload"
    private val path by lazy { requireArguments().getString(ARG_PATH)!! }
    private val binding by viewBinding(FragmentSelfieUploadBinding::bind)
    private val selfieUploadVm by viewModel<SelfieUploadVm> {
        parametersOf(path)
    }

    private val retryCount get() = requireArguments().getInt(ARG_RETRY_COUNT, 0)

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        binding.mainIvConfirmSelfieFrag.setImageURI(Uri.fromFile(File(path)))
        setUpObserver()
    }

    override fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.changeTheme(MatiToolbar.Theme.NONE)
    }

    private fun incrementRetryCount() = requireArguments().putInt(ARG_RETRY_COUNT, retryCount + 1)


    private fun setUpObserver() {
        viewLifecycleOwner.lifecycleScope.launch {
            selfieUploadVm.state.collect {
                when (it) {
                    is SelfieUploadVm.State.Loading -> {}
                    is SelfieUploadVm.State.Success -> {
                        navigation.openNextStep()
                        track(BiometryUploadEvent(Processed(), BiometryType.SELFIE))
                    }
                    is SelfieUploadVm.State.Error -> {
                        val (iconId, error, errorDescription) =  MediaVerificationError.byId(it.error.type.id).let { mediaVerificationError ->
                            Triple( mediaVerificationError.iconId,
                                requireContext().getString(mediaVerificationError.title),
                                requireContext().getString(mediaVerificationError.subtitle)
                            )
                        }
                        val destination = BaseErrorFragment.destination(
                            prepareMediaProcessingErrorScreenData(iconId,error,errorDescription,getString(R.string.label_try_again))
                        )
                        handleError(destination, it.error.type.id)
                    }
                }
            }
        }
    }

    private fun handleError(destination: MatiDestination, code: String, message: String? = null) {
        track(BiometryUploadEvent(Failed(code, message, retryCount), BiometryType.SELFIE))
        incrementRetryCount()
        navigation.back()
        navigation.navigateTo(destination)
    }

    private fun getErrorScreenData(mErrorCode: Int, context: Context) = when (mErrorCode) {
        ErrorScreenConstants.ERROR_CODE_NO_INTERNET_CONNECTION, HttpURLConnection.HTTP_CLIENT_TIMEOUT ->
            prepareBaseErrorScreenData(
            R.drawable.ic_error_other,
            context.getString(R.string.label_something_went_wrong),
            context.getString(R.string.label_check_your_internet),
            context.getString(R.string.label_try_again),
        )
        else -> prepareBaseErrorScreenData(
            R.drawable.ic_error_other,
            context.getString(R.string.label_something_went_wrong),
            context.getString(R.string.label_unexpected_error),
            context.getString(R.string.label_try_again)
        )
    }

    companion object {

        private const val ARG_PATH = "ARG_PATH"
        private const val ARG_RETRY_COUNT = "ARG_RETRY_COUNT"

        fun destination(pPhotoPath: String) = MatiDestination(
            R.id.to_selfieUpload,
            Bundle().apply {
                putString(ARG_PATH, pPhotoPath)
            }
        )
    }
}
