package com.getmati.mati_sdk.ui.start

import android.content.Intent
import android.graphics.Color
import android.net.Uri
import android.os.Bundle
import android.text.SpannableString
import android.text.Spanned
import android.text.method.LinkMovementMethod
import android.text.style.ClickableSpan
import android.view.View
import android.widget.TextView
import androidx.activity.OnBackPressedCallback
import androidx.core.os.bundleOf
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Clicked
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.events.Verification
import com.getmati.mati_sdk.analytics.events.VerificationStarted
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentStartVerificationBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataHolder
import com.getmati.mati_sdk.models.clean.verification.biometryType
import com.getmati.mati_sdk.models.clean.verification.documentUploadSteps
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.phonevalidation.AttemptsExhaustedFragment
import com.getmati.mati_sdk.ui.utils.LocaleManager
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import org.koin.android.ext.android.inject

internal class StartVerificationFragment : KYCBaseFragment(R.layout.fragment_start_verification) {

    override val screenName = "verificationStart"

    private val binding by viewBinding(FragmentStartVerificationBinding::bind)
    private val prefetchDataHolder by inject<PrefetchDataHolder>()

    override fun configureToolbar(toolbar: MatiToolbar) {
        super.configureToolbar(toolbar)
        toolbar.setLanguage(LocaleManager.getLanguageId())
        verificationFlow.logoUrl?.let { toolbar.setLogo(it) }
        toolbar.setChooseLanguageVisible(prefetchDataHolder.prefetchedData.config?.fixedLanguage == null)
        toolbar.setBackImageVisibile(false)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        verificationFlow.apply {
            verificationSteps.firstOrNull()?.let {
                track(
                    Verification(
                        VerificationStarted(),
                        it,
                        documentUploadSteps.map { group -> group.map { it.id } },
                        verificationSteps.size,
                        biometryType
                    )
                )
            }
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        if (verificationFlow.verificationSteps.isEmpty()) {
            navigation.navigateTo(
                AttemptsExhaustedFragment.destination(
                    AttemptsExhaustedFragment.Error.INVALID_CONFIGURATION
                )
            )
            return
        }

        binding.verificationStepsRV.run {
            adapter = VerificationStepDescriptionAdapter(verificationFlow)
            (layoutManager as LinearLayoutManager).orientation =
                if (verificationFlow.verificationSteps.map { it::class }
                        .toSet().size == 2) RecyclerView.HORIZONTAL else RecyclerView.VERTICAL
        }

        setupTerms()

        binding.actionPrimary.setOnClickListener {
            track(UserAction("primaryButton", Clicked(), screenName))
            navigation.startVerification()
        }

        requireActivity().onBackPressedDispatcher.addCallback(
            viewLifecycleOwner,
            object : OnBackPressedCallback(true) {
                override fun handleOnBackPressed() {
                    navigation.cancelVerification()
                }
            })
    }

    private fun setupTerms() {
        val terms: ClickableSpan = object : ClickableSpan() {
            override fun onClick(widget: View) {
                startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(TERMS_AND_CONDITIONS_LINK)))
            }
        }

        val privacy: ClickableSpan = object : ClickableSpan() {
            override fun onClick(widget: View) {
                startActivity(Intent(Intent.ACTION_VIEW, Uri.parse(PRIVACY_POLICY_LINK)))
            }
        }

        val agreementAll = getString(R.string.label_mati_agreement)
        val termsAndConditions = getString(R.string.label_user_terms_and_cond)
        val privacyNotice = getString(R.string.label_mati_privacy_notice)

        setClickableString(
            agreementAll,
            binding.termsAndConditionsTvStartVerificationFrag,
            arrayOf(termsAndConditions, privacyNotice),
            arrayOf(terms, privacy)
        )
    }

    private fun setClickableString(
        wholeValue: String,
        textView: TextView?,
        clickableValue: Array<String>,
        clickableSpans: Array<ClickableSpan>
    ) {
        val spannableString = SpannableString(wholeValue)
        for (i in clickableValue.indices) {
            val clickableSpan = clickableSpans[i]
            val link = clickableValue[i]
            val startIndexOfLink = wholeValue.indexOf(link)
            spannableString.setSpan(
                clickableSpan,
                startIndexOfLink,
                startIndexOfLink + link.length,
                Spanned.SPAN_EXCLUSIVE_EXCLUSIVE
            )
        }
        textView!!.highlightColor = Color.TRANSPARENT
        textView.movementMethod = LinkMovementMethod.getInstance()
        textView.setText(spannableString, TextView.BufferType.SPANNABLE)
    }

    companion object {

        private const val PRIVACY_POLICY_LINK = "https://getmati.com/privacypolicy"
        private const val TERMS_AND_CONDITIONS_LINK = "https://getmati.com/termsofservice"

        fun destination(): MatiDestination {
            return MatiDestination(R.id.startVerificationFragment, bundleOf())
        }
    }
}
