package com.getmati.mati_sdk.ui.utils

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Color
import android.graphics.Matrix
import android.graphics.pdf.PdfRenderer
import android.media.ExifInterface
import android.os.ParcelFileDescriptor
import android.widget.ImageView
import androidx.annotation.DrawableRes
import androidx.core.content.FileProvider
import coil.load
import com.getmati.mati_sdk.Constants.EXTENSION_JPG
import com.getmati.mati_sdk.Constants.TEMP_FILE_NAME
import java.io.File
import java.io.FileNotFoundException
import java.io.FileOutputStream
import java.text.SimpleDateFormat
import java.util.*
import kotlin.math.max
import kotlin.math.min

internal const val MIN_DIMENSION = 600

internal fun Context.getImagePath(fileName: String): String {
    return "$cacheDir/$fileName$EXTENSION_JPG"
}

internal fun Context.createTempFile(
    fileName: String = TEMP_FILE_NAME,
    extension: String = EXTENSION_JPG
) =
    File("$cacheDir/$fileName$extension")

internal fun Context.createTempFileUri(
    fileName: String = TEMP_FILE_NAME,
    extension: String = EXTENSION_JPG
) = FileProvider.getUriForFile(
    this,
    "${applicationContext.packageName}.provider",
    createTempFile(fileName, extension)
)


internal fun rotateAndScaleDown(srcPath: String): BitmapTransformation {
    val orientation: Int = ExifInterface(srcPath).getAttributeInt(
        ExifInterface.TAG_ORIENTATION,
        ExifInterface.ORIENTATION_UNDEFINED
    )

    val sampleSize = BitmapFactory.Options().let {
        it.inJustDecodeBounds = true
        BitmapFactory.decodeFile(srcPath, it)
        min(it.outHeight, it.outWidth) / MIN_DIMENSION
    }
    return if (sampleSize >= 1) {
        BitmapFactory.decodeFile(srcPath, BitmapFactory.Options().apply {
            inSampleSize = sampleSize
        })?.let {
            BitmapTransformation.Success(it.rotateAndScaleDown(orientation))
        } ?: BitmapTransformation.BitmapDecodingError
    } else {
        BitmapTransformation.SmallImageError
    }
}

internal fun Bitmap.rotateAndScaleDown(orientation: Int): Bitmap {
    val matrix = Matrix().apply {
        val scale = MIN_DIMENSION.toFloat() / min(height, width)
        setScale(scale, scale)

        when (orientation) {
            ExifInterface.ORIENTATION_ROTATE_90 -> setRotate(90f)
            ExifInterface.ORIENTATION_ROTATE_180 -> setRotate(180f)
            ExifInterface.ORIENTATION_ROTATE_270 -> setRotate(-90f)
            ExifInterface.ORIENTATION_FLIP_HORIZONTAL -> setScale(-1f, 1f)
            ExifInterface.ORIENTATION_FLIP_VERTICAL -> {
                setRotate(180f)
                postScale(-1f, 1f)
            }
            ExifInterface.ORIENTATION_TRANSPOSE -> {
                setRotate(90f)
                postScale(-1f, 1f)
            }
            ExifInterface.ORIENTATION_TRANSVERSE -> {
                setRotate(-90f)
                postScale(-1f, 1f)
            }
        }
    }

    val result = Bitmap.createBitmap(this, 0, 0, width, height, matrix, true)

    if (this != result) recycle()

    return result
}


internal fun Context.tryConvertPdfToImage(file: File): PdfToImageTransformation {
    if (file.extension != "pdf") {
        return PdfToImageTransformation.NotPdfFile(file.path)
    }
    val fileDescriptor = getSeekableFileDescriptor(file)
        ?: return PdfToImageTransformation.PdfTransformationError

    val timeStampPattern = "yyyyMMdd_HHmmss"

    try {
        val renderer = PdfRenderer(fileDescriptor)
        val page = renderer.openPage(0)

        //Creating bitmap
        val scaleAmount = max(
            max(this.resources.displayMetrics.widthPixels, MIN_DIMENSION) / page.width,
            max(this.resources.displayMetrics.heightPixels, MIN_DIMENSION) / page.height
        )
        val width = page.width * scaleAmount
        val height = page.height * scaleAmount
        val mBitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888)
        mBitmap.eraseColor(Color.WHITE)

        //Rendering bitmap from page
        page?.render(mBitmap, null, null, PdfRenderer.Page.RENDER_MODE_FOR_DISPLAY)

        //Storing the bitmap
        val timeStamp = SimpleDateFormat(timeStampPattern).format(Date())
        val newPathFile = this.getImagePath("${timeStamp}_pdf.jpg")
        val os = FileOutputStream(newPathFile)
        mBitmap.compress(Bitmap.CompressFormat.JPEG, 100, os)

        os.flush()
        os.close()
        page?.close()
        renderer.close()
        return PdfToImageTransformation.Success(newPathFile)
    } catch (e: Exception) {
        return PdfToImageTransformation.PdfTransformationError
    }
}

private fun getSeekableFileDescriptor(file: File): ParcelFileDescriptor? {
    var fd: ParcelFileDescriptor? = null
    try {
        fd = ParcelFileDescriptor.open(file, ParcelFileDescriptor.MODE_READ_ONLY)
    } catch (e: FileNotFoundException) {
        e.printStackTrace()
    }
    return fd
}

internal inline fun ImageView.loadRemoteImage(url: String, crossinline onSuccess: () -> Unit = {}) {
    load(url) {
        target(onSuccess = {
            setImageDrawable(it)
            onSuccess()
        })
    }
}

internal fun ImageView.loadImageResource(@DrawableRes resId: Int) {
    load(resId)
}

internal fun ImageView.loadImageFromFileUrl(url: String) {
    load(File(url))
}

sealed class BitmapTransformation {
    data class Success(val bitmap: Bitmap) : BitmapTransformation()
    object SmallImageError : BitmapTransformation()
    object BitmapDecodingError : BitmapTransformation()
}

sealed class PdfToImageTransformation {
    data class Success(val pathToPdf: String) : PdfToImageTransformation()
    data class NotPdfFile(val pathToNonPdf: String) : PdfToImageTransformation()
    object PdfTransformationError : PdfToImageTransformation()
}