package com.getmati.mati_sdk.ui.verification

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.getmati.mati_sdk.BuildConfig
import com.getmati.mati_sdk.analytics.AnalyticsRegistry
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataHolder
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataSaveStateHandler
import com.getmati.mati_sdk.managers.reporter.CrashReporter
import com.getmati.mati_sdk.models.clean.SocketEvent
import com.getmati.mati_sdk.models.clean.input.Input
import com.getmati.mati_sdk.models.clean.input.InputError
import com.getmati.mati_sdk.models.clean.input.InputStatus
import com.getmati.mati_sdk.models.clean.verification.VerificationFlow
import com.getmati.mati_sdk.server.SocketManager
import com.getmati.mati_sdk.ui.data_prefetch.model.PrefetchedData
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.core.component.KoinComponent
import org.koin.core.component.inject

internal class VerificationVm(private val analyticsRegistry: AnalyticsRegistry,
                              private val savedStateHandle: SavedStateHandle,
                              private val prefetchDataSaveStateHandler: PrefetchDataSaveStateHandler,
                              private val prefetchDataHolder: PrefetchDataHolder,
                              private val socketManager: SocketManager) : ViewModel(), KoinComponent {

    private val reporter: CrashReporter by inject()

    private val mutableInputsFlow by lazy { MutableStateFlow(prefetchDataHolder.initialInputs) }

    val verificationFlow: VerificationFlow by lazy { prefetchDataHolder.verificationFlow }
    val initialInputs: List<Input> by lazy { prefetchDataHolder.initialInputs }
    val prefetchedData: PrefetchedData by lazy { prefetchDataHolder.prefetchedData }
    val inputsFlow: StateFlow<List<Input>>
        get() = mutableInputsFlow

    init {
        registerComponents()

        viewModelScope.launch {
            socketManager.eventFlow.collect { event ->
                if (event is SocketEvent.VerificationResultEvent) {
                    val verificationResult = event.verificationResult
                    if (verificationResult.verificationDataResult?.inputId == "connection-data") {
                        val error = verificationResult.verificationResultError
                        val input = if (error != null) {
                            Input(
                                id = "connection-data",
                                status = InputStatus.ERROR,
                                error = InputError(error.code, verificationResult.optional ?: false)
                            )
                        } else {
                            Input(id = "connection-data", status = InputStatus.SUCCESS)
                        }

                        val newInputs =
                            mutableInputsFlow.value.map { if (it.id == "connection-data") input else it }
                        mutableInputsFlow.value = newInputs
                    }
                }
                if (event is SocketEvent.RoomJoined) {
                    mutableInputsFlow.value = event.inputs
                }
            }
        }
    }

    private fun registerComponents() {
        prefetchDataSaveStateHandler.handle(prefetchDataHolder, savedStateHandle)
        analyticsRegistry.initAnalytics()
        if (!BuildConfig.DEBUG) {
            reporter.registerUncaughtExceptionHandler()
        }
        if (!socketManager.connected()) {
            socketManager.connect()
        }
    }

    override fun onCleared() {
        if (!BuildConfig.DEBUG) {
            reporter.unregisterUncaughtExceptionHandler()
        }
        socketManager.disconnect()
        prefetchDataHolder.clearPrefetchData()
        super.onCleared()
    }
}

