package com.getmati.mati_sdk.ui.web

import android.annotation.SuppressLint
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.activity.OnBackPressedCallback
import androidx.core.view.isVisible
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.FragmentWebVerificationBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding

import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.ui.common.ExitFragment
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import com.getmati.mati_sdk.analytics.events.Failed
import com.getmati.mati_sdk.analytics.events.Loaded
import com.getmati.mati_sdk.analytics.events.WebVerificationEvent
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import kotlinx.coroutines.flow.collect
import org.koin.androidx.viewmodel.ext.android.viewModel
import org.koin.core.parameter.parametersOf
import java.net.HttpURLConnection


internal class WebVerificationFragment : KYCBaseFragment(R.layout.fragment_web_verification) {
    override val screenName = "webVerification"
    private val viewModel by viewModel<WebVerificationVm> {
        parametersOf(webVerificationType)
    }
    private val binding by viewBinding(FragmentWebVerificationBinding::bind)
    private val webVerificationType by lazy { requireArguments().getString(ARG_PATH)!! }
    private var loadedView: View? = null
    private var isLoaded = false

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        if (loadedView == null) {
            loadedView = inflater.inflate(R.layout.fragment_web_verification, container, false);

        }
        return loadedView
    }

    @SuppressLint("SetJavaScriptEnabled")
    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        viewModel.resetState()
        setUpBackPressBehaviour()
        setUpObserver()
        if (!isLoaded) {
            isLoaded = true
            binding.webView.settings.javaScriptEnabled = true
            binding.webView.addJavascriptInterface(
                AndroidBridge(viewModel.WebCallback(), webVerificationType),
                CLASS_NAME_FOR_WEB
            )
            binding.webView.loadUrl(viewModel.createWebUrl())
            binding.webView.webViewClient = object : WebViewClient() {
                override fun onPageFinished(view: WebView?, url: String?) {
                    super.onPageFinished(view, url)
                    track(
                        WebVerificationEvent(
                            webVerificationType,
                            Loaded()
                        )
                    )
                }

            }
        }

    }

    private fun setUpObserver() {
        viewLifecycleOwner.lifecycleScope.launch {
            viewModel.state.collect {
                when (it) {
                    is WebVerificationVm.State.Loading -> {
                        binding.progress.visibility = View.VISIBLE
                    }
                    is WebVerificationVm.State.Success -> {
                        navigation.openNextStep()
                    }
                    is WebVerificationVm.State.HandShake -> {
                        binding.progress.visibility = View.INVISIBLE
                        viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Main) {
                            binding.webView.evaluateJavascript(
                                "javascript: " + "window.dispatchEvent(new CustomEvent('message', { detail: {type: '" + "Handshake" + "'  , payload: " + "undefined" + "}}));",
                                null
                            )
                        }
                    }
                    is WebVerificationVm.State.ErrorTimeOut -> {
                        binding.progress.visibility = View.INVISIBLE
                        track(
                            WebVerificationEvent(
                                "",
                                Failed(
                                    HttpURLConnection.HTTP_CLIENT_TIMEOUT.toString(),
                                    "Web Url loading Timeout",
                                    0
                                )
                            )
                        )
                        navigation.navigateTo(
                            BaseErrorFragment.destination(
                                prepareMediaErrorScreenData(
                                    title = getString(R.string.label_check_your_internet),
                                    primaryCTALabel = getString(R.string.label_try_again)
                                )
                            )
                        )
                    }
                    is WebVerificationVm.State.Interrupt -> {
                        binding.progress.visibility = View.INVISIBLE
                        navigation.navigateTo(ExitFragment.destination())
                    }
                    is WebVerificationVm.State.Skip -> {
                        binding.progress.visibility = View.INVISIBLE
                        navigation.openNextStep()
                    }
                    is WebVerificationVm.State.Error -> {
                        binding.progress.visibility = View.INVISIBLE
                        navigation.navigateTo(
                            BaseErrorFragment.destination(
                                prepareMediaErrorScreenData(
                                    title = getString(R.string.label_something_went_wrong),
                                    primaryCTALabel = getString(R.string.label_try_again)
                                )
                            )
                        )
                    }
                }
            }
        }
    }

    override fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.visibility = View.GONE
    }

    private fun setUpBackPressBehaviour() {
        requireActivity().onBackPressedDispatcher.addCallback(
            viewLifecycleOwner,
            object : OnBackPressedCallback(true) {
                override fun handleOnBackPressed() {
                    if (binding.webView.canGoBack()) {
                        binding.webView.goBack()
                    } else {
                        navigation.navigateTo(ExitFragment.destination())
                    }
                }
            })
    }

    companion object {
        private const val ARG_PATH = "ARG_PATH"

        private const val CLASS_NAME_FOR_WEB = "AndroidBridge"

        fun destination(webVerificationType: String): MatiDestination {
            return MatiDestination(R.id.to_web_verification,
                Bundle().apply {
                    putString(ARG_PATH, webVerificationType)
                })
        }
    }
}