package com.getmati.mati_sdk.ui.web

import android.net.Uri
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.getmati.mati_sdk.logger.LoggerFactory
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataHolder
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import java.util.*
import androidx.annotation.ColorInt
import com.getmati.mati_sdk.ui.utils.LocaleManager


internal class WebVerificationVm(
    private val type: String,
    private val prefetchDataHolder: PrefetchDataHolder,
    private val webVerificationBaseUrl: String,
) : ViewModel() {

    private val _state = MutableStateFlow<State>(State.Loading)
    val state: StateFlow<State> get() = _state
    private var isPageLoaded: Boolean = false

    init {
        scheduleTimerOnLoading()
    }

    fun createWebUrl(): String {
        val prefetchedData = prefetchDataHolder.prefetchedData

        val builtUri =
            Uri.parse(Uri.decode("$WEB_URL_PARAM_SCHEME$webVerificationBaseUrl${prefetchDataHolder.webContainerConfig.first { it.inputID == type }.retrievingURI}"))
                .buildUpon()
                .appendQueryParameter(WEB_URL_PARAM_MOBILE, "true")
                .appendQueryParameter(WEB_URL_PARAM_COLOR, getMatiColorInWebContainerFormat())
                .appendQueryParameter(
                    WEB_URL_PARAM_LOCALE,
                    prefetchDataHolder.config?.fixedLanguage ?: LocaleManager.getLanguageId()
                )
                .appendQueryParameter(WEB_URL_PARAM_CLIENT_ID, prefetchedData.clientId)
                .appendQueryParameter(WEB_URL_PARAM_FLOW_ID, prefetchDataHolder.verificationFlow.id)
                .appendQueryParameter(WEB_URL_PARAM_IDENTITY_ID, prefetchedData.identityId)
                .appendQueryParameter(WEB_URL_PARAM_VERIFICATION_ID, prefetchedData.verificationId)
                .build()

        return builtUri.toString()
    }

    private fun scheduleTimerOnLoading() {
        viewModelScope.launch {
            delay(PAGE_LOADING_TIMEOUT)
            if (!isPageLoaded) {
                _state.value = State.ErrorTimeOut
            }
        }
    }

    fun resetState() {
        _state.value = State.Loading
    }

    private fun getMatiColorInWebContainerFormat(): String? {
        @ColorInt val color = prefetchDataHolder.config?.buttonColor
        return if (color != null) String.format("%06x", color and 0xffffff) else null

    }

    inner class WebCallback {
        fun onHandShake() {
            _state.value = State.HandShake
            isPageLoaded = true
        }

        fun onSuccess() {
            _state.value = State.Success
        }

        fun onInterrupt() {
            _state.value = State.Interrupt
        }

        fun onFailure() {
            _state.value = State.Error
        }

        fun onSkip() {
            _state.value = State.Skip
        }

    }


    sealed class State {
        object Loading : State()
        object Success : State()
        object HandShake : State()
        object ErrorTimeOut : State()
        object Skip : State()
        object Interrupt : State()
        object Error : State()
    }

    companion object {
        private const val PAGE_LOADING_TIMEOUT = 30000L


        private const val WEB_URL_PARAM_SCHEME = "https://"
        private const val WEB_URL_PARAM_TYPE = "type"
        private const val WEB_URL_PARAM_MOBILE = "mobile"
        private const val WEB_URL_PARAM_COLOR = "color"
        private const val WEB_URL_PARAM_LOCALE = "locale"
        private const val WEB_URL_PARAM_CLIENT_ID = "merchantToken"
        private const val WEB_URL_PARAM_FLOW_ID = "flowId"
        private const val WEB_URL_PARAM_IDENTITY_ID = "identityId"
        private const val WEB_URL_PARAM_VERIFICATION_ID = "verificationId"
    }

}