package com.getmati.mati_sdk.widgets

import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import androidx.constraintlayout.widget.ConstraintLayout
import android.view.View
import android.widget.EditText
import androidx.core.content.ContextCompat
import androidx.core.widget.doAfterTextChanged
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.CpfEditTextBinding
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow


class CpfEditText @JvmOverloads
constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) : ConstraintLayout(context, attrs, defStyleAttr) {

    private val _state = MutableStateFlow<State>(State.Empty)
    val state: StateFlow<State> = _state

    private val binding: CpfEditTextBinding
    private var editTextsList: ArrayList<EditText>

    val text: String
        get() =
            "${binding.et1.text}.${binding.et2.text}.${binding.et3.text}-${binding.et4.text}"

    init {
        LayoutInflater.from(context).inflate(R.layout.cpf_edit_text, this, true)
        binding = CpfEditTextBinding.bind(this)
        editTextsList = arrayListOf(binding.et1, binding.et2, binding.et3, binding.et4)
        configureJumps(binding.et1, binding.et2)
        configureJumps(binding.et2, binding.et3, binding.et1)
        configureJumps(binding.et3, binding.et4, binding.et2)
        configureJumps(binding.et4, null, binding.et3)
    }

    private fun configureJumps(
        etFrom: EditText,
        etTo: EditText? = null,
        etBefore: EditText? = null,
        maxDigits: Int = 3
    ) {
        etFrom.doAfterTextChanged { text ->
            if (text?.length == maxDigits) {
                etTo?.requestFocus()
            } else if (text?.length == 0) {
                etBefore?.requestFocus()
            }
            updateState()
        }
    }

    private fun updateState() {
        var enteredCpfDigitsCount = 0
        editTextsList.forEach { enteredCpfDigitsCount += it.text.length }
        _state.value = when (enteredCpfDigitsCount) {
            0 -> {
                hideError()
                State.Empty
            }
            CPF_NUMBER_LENGTH -> {
                if (isDummyData() || !isValidCPFNumber()) {
                    showError()
                    State.NotComplete
                } else {
                    hideError()
                    State.Filled
                }
            }
            else -> {
                hideError()
                State.NotComplete
            }
        }
    }

    private fun isValidCPFNumber(): Boolean {
        val cpfArray = ArrayList<Int>()
        editTextsList.forEach {
            it.text.toString().toCharArray().forEach {
                cpfArray.add(it.toString().toInt())
            }
        }

        var firstCheckSum = 0
        for (i in 0..8) {
            firstCheckSum += cpfArray.get(i) * (10 - i);
        }
        if (((firstCheckSum * 10) % 11) % 10 != cpfArray.get(9)) {
            return false;
        }

        var secondCheckSum = 0
        for (i in 0..9) {
            secondCheckSum += cpfArray[i] * (11 - i);
        }
        return ((secondCheckSum * 10) % 11) % 10 == cpfArray[10];
    }

    //Validating dummy inputs like 1111111 or 00000000 etc.
    private fun isDummyData(): Boolean {
        val set = HashSet<Char>()
        editTextsList.forEach {
            it.text.toString().toCharArray().forEach {
                set.add(it)
            }
        }
        return set.size == 1
    }

    private fun showError() {
        binding.textError.visibility = View.VISIBLE
        editTextsList.forEach {
            it.background =
                ContextCompat.getDrawable(binding.et1.context, R.drawable.bg_et_cpf_error)
        }
    }


    private fun hideError() {
        binding.textError.visibility = View.INVISIBLE
        editTextsList.forEach {
            it.background =
                ContextCompat.getDrawable(binding.et1.context, R.drawable.bg_et_cpf_normal)
        }
    }

    sealed class State {
        object Empty : State()
        object Filled : State()
        object NotComplete : State()
    }

    companion object {
        const val CPF_NUMBER_LENGTH = 11
    }
}