package com.getmati.mati_sdk.widgets

import android.content.Context
import android.graphics.*
import android.util.AttributeSet
import android.view.View
import androidx.vectordrawable.graphics.drawable.VectorDrawableCompat
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.ui.utils.Util.dpToPx
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.drawable.ColorDrawable
import android.graphics.drawable.Drawable
import kotlin.math.roundToInt

internal class DocumentCameraOverlay : View {
    constructor(context: Context?) : super(context!!)
    constructor(context: Context?, attrs: AttributeSet?) : super(context!!, attrs)
    constructor(context: Context?, attrs: AttributeSet?, defStyleAttr: Int) : super(context!!, attrs, defStyleAttr)

    private val aspectRatio = 374f / 264f // From R.drawable.bg_green_corner_large

    private val compensateForStrokeWidth = 2000f / 2 - 1

    private val margin by lazy { dpToPx(context, 48f) }
    private val rectWidth by lazy { width - 2 * margin }
    private val cornerRadius by lazy { 13f * rectWidth / 264f + 1000f  }
    private val rectHeight by lazy { (rectWidth * aspectRatio).roundToInt()  }
    private val marginTop by lazy { kotlin.math.max((height - rectHeight.toFloat()) / 2f - 2 * margin, margin / 2f) }

    private val rect by lazy { RectF(
        margin.toFloat() - compensateForStrokeWidth,
        marginTop - compensateForStrokeWidth,
        margin.toFloat() + rectWidth.toFloat() + compensateForStrokeWidth,
        marginTop + rectHeight.toFloat() + compensateForStrokeWidth) }

    private val documentPreviewAspectRatio = 236f / 136f // From R.drawable.ic_dl_camera
    private val documentPreviewImageWidth by lazy { rectWidth - 2 * margin }
    private val documentPreviewImageHeight by lazy { documentPreviewImageWidth * documentPreviewAspectRatio }
    private val documentPreviewImageMarginTop by lazy { (marginTop + (rectHeight - documentPreviewImageHeight) / 2f).roundToInt() }

    var documentPreviewImage: Drawable? = null
        set(value) {
            field = value
            if(width != 0) {
                field?.setBounds(
                    2 * margin,
                    documentPreviewImageMarginTop,
                    width - 2 * margin,
                    documentPreviewImageMarginTop + documentPreviewImageHeight.roundToInt())
                this.invalidate()
            }
        }

    private val greenCornerVector by lazy { VectorDrawableCompat.create(context.resources, R.drawable.bg_green_corner_large, null)?.apply {
        setBounds(margin, marginTop.roundToInt(), margin + rectWidth, marginTop.roundToInt() + rectHeight)
    }}

    private val paint by lazy {
        Paint().apply {
            color = Color.argb(0xC0, 0x00, 0x00, 0x00)
            style = Paint.Style.STROKE
            strokeWidth = 2000f
        }
    }

    override fun onLayout(changed: Boolean, left: Int, top: Int, right: Int, bottom: Int) {
        super.onLayout(changed, left, top, right, bottom)
        documentPreviewImage?.setBounds(
            2 * margin,
            documentPreviewImageMarginTop,
            width - 2 * margin,
            documentPreviewImageMarginTop + documentPreviewImageHeight.roundToInt())
    }

    override fun dispatchDraw(canvas: Canvas) {
        canvas.drawRoundRect(rect, cornerRadius, cornerRadius, paint)
        greenCornerVector?.draw(canvas)
        documentPreviewImage?.draw(canvas)
        super.dispatchDraw(canvas)
    }
}