package com.getmati.mati_sdk.widgets

import android.app.Activity
import android.content.Context
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.TextView
import androidx.core.content.ContextCompat
import coil.load
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Clicked
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.mati_navigation.MatiNavigation
import com.getmati.mati_sdk.models.clean.Language.Companion.fromId
import com.getmati.mati_sdk.ui.common.ExitFragment
import com.getmati.mati_sdk.ui.verification.VerificationActivity
import com.getmati.mati_sdk.ui.start.LanguageSelectFragment
import com.getmati.mati_sdk.ui.utils.loadImageResource


internal class MatiToolbar : FrameLayout {
    private val mCloseImage by lazy { findViewById<ImageView>(R.id.image_close) }
    private val mBackImage by lazy { findViewById<ImageView>(R.id.image_back) }

    private val mIcon by lazy { findViewById<ImageView>(R.id.image_icon) }

    private val mLanguageIcon by lazy { findViewById<ImageView>(R.id.image_language) }
    private val mLanguageLabel by lazy { findViewById<TextView>(R.id.label_language) }

    var defaultLogo: Boolean = true
    var mTheme: Theme = Theme.UNDEFINED

    val navigation: MatiNavigation?
        get() {
            return (context as? VerificationActivity)?.navigation
        }

    constructor(context: Context?) : super(context!!, null)
    constructor(context: Context?, attrs: AttributeSet?) : super(context!!, attrs) {
        LayoutInflater.from(context)
            .inflate(R.layout.widget_mati_toolbar, this, true)

        listOf(mLanguageIcon, mLanguageLabel).forEach {
            it.setOnClickListener {
                navigation?.navigateTo(LanguageSelectFragment.destination())
            }
        }
        (context as Activity).let {
            mCloseImage.setOnClickListener {
                track(UserAction("exitButton", Clicked(), "verificationExit")) //FIXME tracking !!!
                navigation?.navigateTo(ExitFragment.destination())
            }
            mBackImage.setOnClickListener {
                track(UserAction("backButton", Clicked(), "verificationExit")) //FIXME tracking !!!
                navigation?.back()
            }
        }
    }

    fun changeTheme(theme: Theme) {
        mTheme = theme

        if (defaultLogo) {
            mIcon?.loadImageResource(if (theme == Theme.LIGHT) R.drawable.logo_mati_dark_text_vector else R.drawable.logo_mati_light_text_vector)
        }

        this.setBackgroundColor(
            ContextCompat.getColor(
                context, when (theme) {
                    Theme.DARK -> R.color.matiBackgroundColorDark
                    Theme.VL -> R.color.matiPrimaryText
                    else -> R.color.matiBackgroundColorLight
                }
            )
        )

        this.visibility = if(theme == Theme.NONE) {
            GONE
        } else {
            VISIBLE
        }

        (context as? Activity)?.window?.statusBarColor = ContextCompat.getColor(context, when (theme) {
            Theme.LIGHT -> R.color.design_default_color_on_primary
            else -> R.color.design_default_color_on_secondary
        })

        (context as? Activity)?.window?.decorView?.systemUiVisibility = when (theme) {
            Theme.LIGHT -> View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
            else -> 0
        }
    }

    fun setLogo(url: String) {
        defaultLogo = false
        mIcon?.load(url)
    }

    fun setChooseLanguageVisible(visible: Boolean) {
        listOf(mLanguageIcon, mLanguageLabel).forEach { it.visibility = if(visible) View.VISIBLE else View.GONE }
    }

    fun setBackImageVisibile(visible: Boolean) {
        mBackImage.visibility = if(visible) View.VISIBLE else View.GONE
    }

    fun setCloseImageVisibile(visible: Boolean) {
        mCloseImage.visibility = if(visible) View.VISIBLE else View.GONE
    }

    fun setLanguage(id: String?) {
        val language = fromId(id)
        if (language != null) {
            mLanguageLabel!!.text = context.getString(language.titleRes)
        }
    }

    enum class Theme {
        UNDEFINED,
        NONE,
        LIGHT,
        DARK,
        VL,
    }
}