package com.getmati.mati_sdk.widgets

import android.annotation.SuppressLint
import android.content.*
import android.graphics.Color
import android.text.InputType
import android.util.AttributeSet
import android.view.KeyEvent
import android.view.MotionEvent
import android.view.View
import android.view.ViewGroup
import android.view.inputmethod.BaseInputConnection
import android.view.inputmethod.EditorInfo
import android.view.inputmethod.InputConnection
import android.view.inputmethod.InputMethodManager
import android.widget.LinearLayout
import android.widget.Space
import androidx.annotation.AttrRes
import androidx.annotation.ColorInt
import androidx.annotation.ColorRes
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import androidx.core.view.children
import androidx.core.view.postDelayed
import com.google.android.material.color.MaterialColors

class PassCodeView @JvmOverloads constructor(
        context: Context,
        attrs: AttributeSet? = null,
        defStyleAttr: Int = 0
) : LinearLayout(context, attrs, defStyleAttr) {

    var enteredCode: String = ""
        set(value) {
            require(value.length <= codeLength) { "enteredCode=$value is longer than $codeLength" }
            field = value
            onChangeListener?.onCodeChange(
                    code = value,
                    isComplete = value.length == codeLength
            )
            updateState()
        }

    var onChangeListener: OnChangeListener? = null

    internal var style: Style = PassCodeViewStyleUtils.getDefault(context)
        set(value) {
            if (field == value) return
            field = value
            removeAllViews()
            updateState()
        }

    private val symbolSubviews: Sequence<SymbolView>
        get() = children.filterIsInstance<SymbolView>()

    init {
        orientation = HORIZONTAL
        isFocusable = true
        isFocusableInTouchMode = true

        style =
                if (attrs == null) PassCodeViewStyleUtils.getDefault(context)
                else PassCodeViewStyleUtils.getFromAttributes(attrs, context)

        updateState()

        if (isInEditMode) {
            repeat(codeLength) {
                enteredCode += 0.toString()
            }
        }
    }

    private fun updateState() {
        val codeLengthChanged = codeLength != symbolSubviews.count()
        if (codeLengthChanged) {
            setupSymbolSubviews()
        }

        val viewCode = symbolSubviews.map { it.symbol }
                .filterNotNull()
                .joinToString(separator = "")
        val isViewCodeOutdated = enteredCode != viewCode
        if (isViewCodeOutdated) {
            symbolSubviews.forEachIndexed { index, view ->
                view.symbol = enteredCode.getOrNull(index)
            }
        }
    }

    private fun setupSymbolSubviews() {
        removeAllViews()

        for (i in 0 until codeLength) {
            val symbolView = SymbolView(context, style.symbolViewStyle)
            addView(symbolView)

            if (i < codeLength.dec()) {
                val space = Space(context).apply {
                    layoutParams = ViewGroup.LayoutParams(style.symbolsSpacing, 0)
                }
                addView(space)
            }
        }
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        setOnKeyListener { _, keyCode, event -> handleKeyEvent(keyCode, event) }
        postDelayed(KEYBOARD_AUTO_SHOW_DELAY) {
            requestFocus()
            showKeyboard()
        }
    }

    private fun handleKeyEvent(keyCode: Int, event: KeyEvent): Boolean = when {
        event.action != KeyEvent.ACTION_DOWN -> false
        event.isDigitKey() -> {
            val enteredSymbol = event.keyCharacterMap.getNumber(keyCode)
            appendSymbol(enteredSymbol)
            true
        }
        event.keyCode == KeyEvent.KEYCODE_DEL -> {
            removeLastSymbol()
            true
        }
        event.keyCode == KeyEvent.KEYCODE_ENTER -> {
            hideKeyboard()
            true
        }
        else -> false
    }

    private fun KeyEvent.isDigitKey(): Boolean {
        return this.keyCode in KeyEvent.KEYCODE_0..KeyEvent.KEYCODE_9
    }

    private fun appendSymbol(symbol: Char) {
        if (enteredCode.length == codeLength) {
            return
        }

        this.enteredCode = enteredCode + symbol
    }

    private fun removeLastSymbol() {
        if (enteredCode.isEmpty()) {
            return
        }

        this.enteredCode = enteredCode.substring(0, enteredCode.length - 1)
    }

    @SuppressLint("ClickableViewAccessibility")
    override fun onTouchEvent(event: MotionEvent): Boolean {
        if (event.action == MotionEvent.ACTION_DOWN && requestFocus()) {
            showKeyboard()
        }
        return super.onTouchEvent(event)
    }

    override fun onCheckIsTextEditor(): Boolean = true

    override fun onCreateInputConnection(outAttrs: EditorInfo): InputConnection? {
        with(outAttrs) {
            inputType = InputType.TYPE_CLASS_NUMBER
            imeOptions = EditorInfo.IME_ACTION_DONE
        }

        return BaseInputConnection(this, false)
    }

    fun reset() {
        enteredCode = ""
    }

    fun changeColor(@ColorRes id: Int) {
        val color = ContextCompat.getColor(context, id)
        symbolTextColor = color
        symbolBorderColor = color
    }

    fun interface OnChangeListener {
        fun onCodeChange(code: String, isComplete: Boolean)
    }

    internal data class Style(
            val codeLength: Int,
            val symbolsSpacing: Int,
            val symbolViewStyle: SymbolView.Style
    )

    companion object {
        internal const val DEFAULT_CODE_LENGTH = 4
        private const val KEYBOARD_AUTO_SHOW_DELAY = 500L
    }
}

var PassCodeView.codeLength: Int
    set(value) {
        require(value >= 0) { "invalid code length - $value" }
        this.style = style.copy(codeLength = value)
    }
    get() = style.codeLength

var PassCodeView.symbolsSpacing: Int
    set(value) {
        this.style = style.copy(symbolsSpacing = value)
    }
    get() = style.symbolsSpacing

var PassCodeView.symbolWidth: Int
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                width = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.width

var PassCodeView.symbolHeight: Int
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                height = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.height

var PassCodeView.symbolTextColor: Int
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                textColor = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.textColor

var PassCodeView.symbolFontFamily: Int
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                fontFamily = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.fontFamily

var PassCodeView.symbolTextSize: Int
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                textSize = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.textSize

var PassCodeView.symbolBackgroundColor: Int
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                backgroundColor = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.backgroundColor

var PassCodeView.symbolBorderColor: Int
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                borderColor = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.borderColor

var PassCodeView.symbolBorderWidth: Int
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                borderWidth = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.borderWidth

var PassCodeView.symbolBorderCornerRadius: Float
    set(value) {
        val updatedStyle = style.symbolViewStyle.copy(
                borderCornerRadius = value
        )
        this.style = style.copy(symbolViewStyle = updatedStyle)
    }
    get() = style.symbolViewStyle.borderCornerRadius


@ColorInt
internal fun Context.getThemeColor(@AttrRes attrRes: Int): Int {
    return MaterialColors.getColor(this, attrRes, Color.BLACK)
}

internal fun View.showKeyboard() {
    val imm = context.getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
    imm.showSoftInput(this, InputMethodManager.SHOW_IMPLICIT)
}

internal fun View.hideKeyboard() {
    val imm = context.getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
    imm.hideSoftInputFromWindow(this.windowToken, 0)
}