package net.gdface.utils;

import com.google.common.base.Function;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkArgument;
/**
 * 基于GUAVA {@link LoadingCache}实现类型转转换接口{@link Function},
 * 当一个输入K的计算结果已经存在于缓存时，直接返回，无需再次计算，
 * 避免对同一个输入数据多次重复计算。
 * 适用于输入值与结果恒定对应的场景
 * @author guyadong
 *
 * @param <K> 输入参数类型
 * @param <V> 输出参数类型
 */
public class FunctionCached<K,V> implements Function<K, V> {
	/**
	 * 输入值与计算结果的映射，计算结果可能是正常计算出的V值,也可能是异常(Exception)
	 */
	private final LoadingCache<K, Object> cache;
	private final Function<K, V> getterFunction;
	private final V defaultValue;
	/**
	 * 构造方法
	 * @param getterFunction 原类型转换接口实现
	 * @param defaultValue    K为{@code null}时返回的默认值
	 * @param cacheBuilder    外部提供的CacheBuilder实例,为{@code null}则忽略
	 * @since 2.7.9
	 */
	public FunctionCached(final Function<K, V> getterFunction,V defaultValue,CacheBuilder<Object,Object> cacheBuilder){
		this.getterFunction = checkNotNull(getterFunction,"getterFunction is null");
		/**  输入参数不可以是缓存实例  */
		checkArgument(!(getterFunction instanceof FunctionCached),"getterFunction must not be instance of %s",getClass().getName());
		this.defaultValue = defaultValue;
		if(null == cacheBuilder ) {
			cacheBuilder= CacheBuilder.newBuilder();
		}
		this.cache = cacheBuilder.build(new CacheLoader<K, Object>(){
			@Override
			public Object load(K key) {
				try {
					return getterFunction.apply(key);
				} catch (Exception e) {
					return e;
				}
			}});
	}
	/**
	 * 构造方法
	 * @param getterFunction 原类型转换接口实现
	 * @param defaultValue    K为{@code null}时返回的默认值
	 */
	public FunctionCached(final Function<K, V> getterFunction,V defaultValue){
		this(getterFunction,defaultValue,null);
	}
	/**
	 * 构造方法,
	 * K 为{@code null}时返回{@code null}
	 * @param getterFunction 原类型转换接口实现
	 */
	public FunctionCached(Function<K, V> getterFunction){
		this(getterFunction, null);
	}
	/**
	 * 非缓存调用
	 * @param key
	 * @return {@code key}为{@code null}返回{@link #defaultValue}
	 */
	public V getUncached(K key){
		return null == key ? defaultValue : getterFunction.apply(key);
	}
	/**
	 * 缓存调用
	 * @param key
	 * @return {@code key}为{@code null}返回{@link #defaultValue}
	 */
	@SuppressWarnings("unchecked")
	public V get(K key){
		if(null != key){
			Object value = cache.getUnchecked(key);
			if(value instanceof Exception) {
				Throwables.throwIfUnchecked((Exception)value);
				throw new RuntimeException((Exception)value);
			}
			return (V) value;
		}
		return defaultValue;
	}
	
	/**
	 * 缓存调用
	 * @see #get(Object)
	 * @see com.google.common.base.Function#apply(java.lang.Object)
	 */
	@Override
	public V apply(K input) {
		return get(input);
	}
	/**
	 * 返回{@code getterFunction}的{@link FunctionCached}实例,
	 * 如果{@code getterFunction}为{@link FunctionCached}实例,则直接返回，否则创建新实例
	 * @param getterFunction
	 * @param defaultValue
	 * @param cacheBuilder 外部提供的CacheBuilder实例,为{@code null}则忽略
	 * @see #FunctionCached(Function, Object, CacheBuilder)
	 * @since 2.7.9
	 */
	public static <K, V> FunctionCached<K, V> of(Function<K, V> getterFunction,V defaultValue,CacheBuilder<Object,Object> cacheBuilder){
		if(getterFunction instanceof FunctionCached){
			return (FunctionCached<K, V>) getterFunction;
		}
		return new FunctionCached<K, V>(getterFunction, defaultValue,cacheBuilder); 
	}
	/**
	 * 返回{@code getterFunction}的{@link FunctionCached}实例,
	 * 如果{@code getterFunction}为{@link FunctionCached}实例,则直接返回，否则创建新实例
	 * @param getterFunction
	 * @param defaultValue
	 * @see #of(Function, Object, CacheBuilder)
	 */
	public static <K, V> FunctionCached<K, V> of(Function<K, V> getterFunction,V defaultValue){
		return of(getterFunction, defaultValue,null); 
	}
	/**
	 * 返回{@code getterFunction}的{@link FunctionCached}实例(默认实例为{@code null})
	 * @param getterFunction
	 * @see #of(Function, Object)
	 */
	public static <K, V> FunctionCached<K, V> of(Function<K, V> getterFunction){
		return of(getterFunction,null); 
	}
}
