package com.zzh.lib.pay.alipay;

import android.text.TextUtils;

import com.zzh.lib.pay.L;
import com.zzh.lib.pay.model.HPayInfo;
import com.zzh.lib.pay.model.HPayKey;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @Date: 2022/5/23 15:30
 * @Email: zzh_hz@126.com
 * @QQ: 1299234582
 * @Author: zzh
 * @Description: PayUrlGenerator.java
 */
public class PayUrlGenerator {

    private static final String TAG = "PayUrlGenerator";

    private HPayInfo payInfo;

    public PayUrlGenerator(HPayInfo payInfo) {
        this.payInfo = payInfo;
    }

    public String generatePayUrl() {

        if (this.payInfo == null) {
            L.e(TAG, " +++++ orderInfo is null");
            return "";
        }

        validatePayInfo(payInfo);
        Map<String, String> paramMap = buildOrderParamMap();
        String orderInfo = buildOrderParam(paramMap);
        String sign = genOrderInfo(paramMap);
        L.e("orderInfo: ", orderInfo);
        // 完整的符合支付宝参数规范的订单信息
        StringBuilder sb = new StringBuilder(orderInfo);
        sb.append("&sign=");
        sb.append(sign);

        final String payUrl = sb.toString();
        L.i(TAG, "pay info:" + payUrl);
        return payUrl;
    }

    private String genOrderInfo(Map<String, String> map) {
        List<String> keys = new ArrayList<>(map.keySet());
        // key排序
        Collections.sort(keys);

        StringBuilder authInfo = new StringBuilder();
        for (int i = 0; i < keys.size() - 1; i++) {
            String key = keys.get(i);
            String value = map.get(key);
            authInfo.append(buildKeyValue(key, value, false));
            authInfo.append("&");
        }

        String tailKey = keys.get(keys.size() - 1);
        String tailValue = map.get(tailKey);
        authInfo.append(buildKeyValue(tailKey, tailValue, false));

        String oriSign = signUrl(authInfo.toString());
        String encodedSign = "";

        try {
            encodedSign = URLEncoder.encode(oriSign, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return encodedSign;
    }

    /**
     * 构造支付订单参数信息
     *
     * @param map 支付订单参数
     * @return
     */
    public String buildOrderParam(Map<String, String> map) {
        List<String> keys = new ArrayList<>(map.keySet());

        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < keys.size() - 1; i++) {
            String key = keys.get(i);
            String value = map.get(key);
            sb.append(buildKeyValue(key, value, true));
            sb.append("&");
        }

        String tailKey = keys.get(keys.size() - 1);
        String tailValue = map.get(tailKey);
        sb.append(buildKeyValue(tailKey, tailValue, true));

        return sb.toString();
    }

    /**
     * 拼接键值对
     *
     * @param key
     * @param value
     * @param isEncode
     * @return
     */
    private String buildKeyValue(String key, String value, boolean isEncode) {
        StringBuilder sb = new StringBuilder();
        sb.append(key);
        sb.append("=");
        if (isEncode) {
            try {
                sb.append(URLEncoder.encode(value, "UTF-8"));
            } catch (UnsupportedEncodingException e) {
                sb.append(value);
            }
        } else {
            sb.append(value);
        }
        return sb.toString();
    }

    /**
     * 构造支付订单参数列表
     */
    public Map<String, String> buildOrderParamMap() {
        Map<String, String> keyValues = new HashMap<>();

        String appId = TextUtils.isEmpty(payInfo.getAliAppId()) ? HPayKey.ALI_APP_ID : payInfo.getAliAppId();

        keyValues.put("app_id", appId);

        keyValues.put("biz_content", "{\"timeout_express\":\"" + payInfo.getAliTimeoutExpress() +
                "\",\"product_code\":\"QUICK_MSECURITY_PAY\",\"total_amount\":\"" + payInfo.getPrice() +
                "\",\"subject\":\"" + payInfo.getSubject() +
                "\",\"body\":\"" + payInfo.getBody() +
                "\",\"out_trade_no\":\"" + this.payInfo.getOrderNo() + "\"}");

        keyValues.put("charset", "utf-8");
        keyValues.put("method", "alipay.trade.app.pay");
        keyValues.put("sign_type", "RSA2");
        keyValues.put("notify_url", payInfo.getNotifyUrl());
        String timestamp = TextUtils.isEmpty(payInfo.getTimeStamp()) ? getDateTime() : payInfo.getTimeStamp();
        keyValues.put("timestamp", timestamp);
        keyValues.put("version", "1.0");
        return keyValues;
    }

    private String getDateTime() {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return sdf.format(System.currentTimeMillis());
    }

    // used RSA algorithm
    // 对订单做RSA 签名
    private String signUrl(String urlToSign) {
        try {
            PKCS8EncodedKeySpec priPKCS8 = new PKCS8EncodedKeySpec(Base64.decode(HPayKey.ALI_PRIVATE_KEY));
            KeyFactory keyf = KeyFactory.getInstance("RSA");
            PrivateKey priKey = keyf.generatePrivate(priPKCS8);
            java.security.Signature signature = java.security.Signature.getInstance("SHA256WithRSA");
            signature.initSign(priKey);
            signature.update(urlToSign.getBytes("UTF-8"));
            byte[] signed = signature.sign();
            return Base64.encode(signed);
        } catch (Exception e) {
            L.e(e.getMessage(), e);
        }

        return null;
    }

    /**
     * 验证 支付信息的有效性
     *
     * @return void
     * @autour BaoHong.Li
     * @date 2015-7-16 下午5:56:21
     * @update (date)
     */
    private void validatePayInfo(HPayInfo payInfo) {
        if (TextUtils.isEmpty(payInfo.getOrderNo())) {
            throw new IllegalArgumentException(" payInfo.orderNo is  null !");
        }

        if (TextUtils.isEmpty(payInfo.getAliAppId()) && TextUtils.isEmpty(HPayKey.ALI_APP_ID)) {
            throw new IllegalArgumentException(" payInfo.aliAppId is  null !");
        }

        if (TextUtils.isEmpty(payInfo.getBody())) {
            throw new IllegalArgumentException(" payInfo.body is  null !");
        }

        if (TextUtils.isEmpty(payInfo.getPrice())) {
            throw new IllegalArgumentException(" payInfo.price is  null !");
        }

        if (TextUtils.isEmpty(payInfo.getSubject())) {
            throw new IllegalArgumentException(" payInfo.subject is  null !");
        }
    }
}
