package com.github.bootfastconfig.springtool;

import lombok.Getter;
import org.reflections.Reflections;
import org.reflections.scanners.FieldAnnotationsScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.boot.web.servlet.context.AnnotationConfigServletWebServerApplicationContext;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;

/**
 * @author admin
 * SpringBeanUtil
 * Bean 的工具类
 * 继承它 会自动向子类有
 * <p>
 * Autowired.class
 * Resource.class
 * 注解的实体 注入值 如果有的化
 */

@Component
@Order(Integer.MIN_VALUE)
public class SpringBeanUtil implements ApplicationContextAware {

    private final static Logger log = LoggerFactory.getLogger(SpringBeanUtil.class);

    @Getter
    private static ApplicationContext applicationContext;

    private static BeanDefinitionRegistry beanDefinitionRegistry;

    public SpringBeanUtil() {
        injection(this);
    }

    public static void SpringBeanUtilInit(ApplicationContext applicationContext) {
        if (applicationContext != null) {
            synchronized (SpringBeanUtil.class) {
                SpringBeanUtil.applicationContext = applicationContext;
            }
        }
    }


    /**
     * 手动注入 ioc
     * 默认注解
     * Autowired.class
     * Resource.class
     *
     * @param c
     */
    public static boolean injection(Object c) {
        return injection(c, new Class[]{Autowired.class, Resource.class});
    }

    /**
     * 手动注入
     *
     * @param c
     * @param annotations
     */
    public static boolean injection(Object c, Class<? extends Annotation>[] annotations) {

        if (getApplicationContext() == null || c == null || annotations == null) {
            return false;
        }
        Reflections reflections = new Reflections(c.getClass(), new FieldAnnotationsScanner());
        Set<Field> fields = new HashSet<>();
        Stream.of(annotations).map(o -> {
            return reflections.getFieldsAnnotatedWith(o);
        }).filter(Objects::nonNull).forEach(o -> {
            fields.addAll(o);
        });
        for (Field field : fields) {
            Object bean = getBean(field);
            field.setAccessible(true);
            try {
                field.set(c, bean);
            } catch (IllegalAccessException e) {
                log.error("SpringBeanUtil构造失败", e);
                e.printStackTrace();
            }
        }
        return true;
    }

    private static Object getBean(Field field) {
        String name = field.getName();
        Object bean = null;
        try {
            bean = getApplicationContext().getBean(name);
        } catch (RuntimeException e) {
            log.error("查找bean 失败 未找到 名称为" + name + "的实体", e);
        }
        if (bean != null) {
            return bean;
        }
        return getApplicationContext().getBean(field.getType());
    }

    /**
     * 用bean组件的name来获取bean
     *
     * @param beanName
     */
    @SuppressWarnings("unchecked")
    public static <T> T getBean(String beanName) {
        try {
            return (T) getApplicationContext().getBean(beanName);
        } catch (RuntimeException e) {
            log.error("查找bean 失败 未找到 名称为 [{}] 的实体", beanName, e);
        }
        return null;
    }

    /**
     * 用类来获取bean
     *
     * @param c
     */
    public static <T> T getBean(Class<T> c) {
        return (T) getApplicationContext().getBean(c);
    }

    public static <T> T getBean(String name, Class<T> requiredType) {
        return getApplicationContext().getBean(name, requiredType);
    }


    public static <T> T classInitialization(Class<T> c, String name, Object... constructorValues) {
        if (applicationContext.containsBean(name)) {
            return getBean(name, c);
        }
        BeanDefinition beanDefinition = getBeanDefinition(c, constructorValues);
        return classInitialization(c, name, beanDefinition);
    }

    public static <T> T classInitialization(Class<T> c, String name, BeanDefinition beanDefinition) {
        if (applicationContext.containsBean(name)) {
            return getBean(name, c);
        }
        BeanDefinitionRegistry beanFactory = getBeanDefinitionRegistry(SpringBeanUtil.applicationContext);
        beanFactory.registerBeanDefinition(name, beanDefinition);
        return getBean(name, c);
    }

    public static <T> BeanDefinition getBeanDefinition(Class<T> c, Object... constructorValues) {

        BeanDefinitionBuilder beanDefinitionBuilder = BeanDefinitionBuilder.genericBeanDefinition(c);
        if (constructorValues != null) {
            for (Object constructorValue : constructorValues) {
                beanDefinitionBuilder.addConstructorArgValue(constructorValue);
            }
        }
        beanDefinitionBuilder.setScope(BeanDefinition.SCOPE_PROTOTYPE);
        beanDefinitionBuilder.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_NO);
        return beanDefinitionBuilder.getRawBeanDefinition();
    }

    public static BeanDefinitionRegistry getBeanDefinitionRegistry(ApplicationContext applicationContext) {
        if (beanDefinitionRegistry == null) {
            synchronized (SpringBeanUtil.class) {
                if (beanDefinitionRegistry == null) {
                    if (applicationContext instanceof AnnotationConfigServletWebServerApplicationContext) {
                        beanDefinitionRegistry = (BeanDefinitionRegistry) ((AnnotationConfigServletWebServerApplicationContext) applicationContext).getBeanFactory();
                    }

                }
                if (beanDefinitionRegistry == null) {
                    ConfigurableApplicationContext bean = getBean(ConfigurableApplicationContext.class);
                    beanDefinitionRegistry = (BeanDefinitionRegistry) bean.getBeanFactory();
                }

            }
        }
        return beanDefinitionRegistry;
    }

    public static boolean removeBeanDefinition(String name) {

        BeanDefinitionRegistry beanFactory = getBeanDefinitionRegistry(SpringBeanUtil.applicationContext);
        try {
            beanFactory.removeBeanDefinition(name);
        } catch (Exception e) {
            log.error("删除bean定义失败 bean 名称 [{}]", name, e);
            return false;
        }

        return true;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        SpringBeanUtil.applicationContext = applicationContext;
    }

}
