/*
Copyright 2015 Rudolf Fiala

This file is part of Alpheus AFP Parser.

Alpheus AFP Parser is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Alpheus AFP Parser is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Alpheus AFP Parser.  If not, see <http://www.gnu.org/licenses/>
*/
package com.mgz.afp.modca;

import com.mgz.afp.base.StructuredFieldBaseTriplets;
import com.mgz.afp.enums.AFPUnitBase;
import com.mgz.afp.exceptions.AFPParserException;
import com.mgz.afp.parser.AFPParserConfiguration;
import com.mgz.afp.triplets.Triplet;
import com.mgz.util.UtilBinaryDecoding;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * MO:DCA, page 241.<br> <br> The Medium Descriptor structured field specifies the size and
 * orientation of the medium presentation space for all sheets that are generated by the medium map
 * that contains the Medium Descriptor structured field.
 */
public class MDD_MediumDescriptor extends StructuredFieldBaseTriplets {
  AFPUnitBase xUnitBase;
  AFPUnitBase yUnitBase;
  short xUnitsPerUnitBase;
  short yUnitsPerUnitBase;
  int xMediumExtent;
  int yMediumExtent;
  MDD_Flag flag;

  @Override
  public void decodeAFP(byte[] sfData, int offset, int length, AFPParserConfiguration config) throws AFPParserException {
    xUnitBase = AFPUnitBase.valueOf(sfData[offset]);
    yUnitBase = AFPUnitBase.valueOf(sfData[offset + 1]);
    xUnitsPerUnitBase = UtilBinaryDecoding.parseShort(sfData, offset + 2, 2);
    yUnitsPerUnitBase = UtilBinaryDecoding.parseShort(sfData, offset + 4, 2);
    xMediumExtent = UtilBinaryDecoding.parseInt(sfData, offset + 6, 3);
    yMediumExtent = UtilBinaryDecoding.parseInt(sfData, offset + 9, 3);
    flag = MDD_Flag.valueOf(sfData[offset + 12]);

    int actualLength = getActualLength(sfData, offset, length);
    if (actualLength > 13) {
      super.decodeAFP(sfData, offset + 13, actualLength - 13, config);
    } else {
      triplets = null;
    }
  }

  @Override
  public void writeAFP(OutputStream os, AFPParserConfiguration config) throws IOException {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    baos.write(xUnitBase.toByte());
    baos.write(yUnitBase.toByte());
    baos.write(UtilBinaryDecoding.shortToByteArray(xUnitsPerUnitBase, 2));
    baos.write(UtilBinaryDecoding.shortToByteArray(yUnitsPerUnitBase, 2));
    baos.write(UtilBinaryDecoding.intToByteArray(xMediumExtent, 3));
    baos.write(UtilBinaryDecoding.intToByteArray(yMediumExtent, 3));
    baos.write(flag.toByte());
    if (triplets != null) {
      for (Triplet t : triplets) {
        t.writeAFP(baos, config);
      }
    }

    writeFullStructuredField(os, baos.toByteArray());
  }

  public AFPUnitBase getxUnitBase() {
    return xUnitBase;
  }

  public void setxUnitBase(AFPUnitBase xUnitBase) {
    this.xUnitBase = xUnitBase;
  }

  public AFPUnitBase getyUnitBase() {
    return yUnitBase;
  }

  public void setyUnitBase(AFPUnitBase yUnitBase) {
    this.yUnitBase = yUnitBase;
  }

  public short getxUnitsPerUnitBase() {
    return xUnitsPerUnitBase;
  }

  public void setxUnitsPerUnitBase(short xUnitsPerUnitBase) {
    this.xUnitsPerUnitBase = xUnitsPerUnitBase;
  }

  public short getyUnitsPerUnitBase() {
    return yUnitsPerUnitBase;
  }

  public void setyUnitsPerUnitBase(short yUnitsPerUnitBase) {
    this.yUnitsPerUnitBase = yUnitsPerUnitBase;
  }

  public int getxMediumExtent() {
    return xMediumExtent;
  }

  public void setxMediumExtent(int xMediumExtent) {
    this.xMediumExtent = xMediumExtent;
  }

  public int getyMediumExtent() {
    return yMediumExtent;
  }

  public void setyMediumExtent(int yMediumExtent) {
    this.yMediumExtent = yMediumExtent;
  }

  public MDD_Flag getFlag() {
    return flag;
  }

  public void setFlag(MDD_Flag flag) {
    this.flag = flag;
  }

  public enum MDD_Flag {
    DoNotPassMediumOrientationToCutsheetPrinter,
    PassMediumOrientationToCutsheetPrinter;

    public static MDD_Flag valueOf(byte codeByte) {
      if ((codeByte & 0x80) == 0) {
        return DoNotPassMediumOrientationToCutsheetPrinter;
      } else {
        return PassMediumOrientationToCutsheetPrinter;
      }
    }

    public int toByte() {
      if (this == DoNotPassMediumOrientationToCutsheetPrinter) {
        return 0x00;
      } else {
        return 0x80;
      }
    }
  }
}
