package com.github.aidensuen.mongo.spring;

import com.github.aidensuen.mongo.annotation.RegisterMongoDao;
import com.github.aidensuen.mongo.builder.resource.MongoDaoResourceBuilderAdapter;
import com.github.aidensuen.mongo.plugin.Interceptor;
import com.github.aidensuen.mongo.session.Configuration;
import com.github.aidensuen.mongo.session.MongoSessionFactory;
import com.github.aidensuen.mongo.session.impl.DefaultMongoSessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.type.classreading.CachingMetadataReaderFactory;
import org.springframework.core.type.classreading.MetadataReaderFactory;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.util.ObjectUtils;

import static org.springframework.util.Assert.notNull;

public class MongoSessionFactoryBean implements FactoryBean<MongoSessionFactory>, InitializingBean, ApplicationListener<ApplicationEvent> {

    private static final Logger LOGGER = LoggerFactory.getLogger(MongoSessionFactoryBean.class);

    private ResourceLoader resourceLoader;

    private Configuration configuration;

    private MongoOperations mongoOperations;

    private Interceptor[] plugins;

    private MongoSessionFactory mongoSessionFactory;

    private Resource[] mongodaoLocations;

    private String registerMongoDaoName = RegisterMongoDao.class.getName();

    private MongoDaoResourceBuilderAdapter mongoDaoResourceBuilderAdapter;

    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    public void setMongoOperations(MongoOperations mongoOperations) {
        this.mongoOperations = mongoOperations;
    }

    public void setPlugins(Interceptor[] plugins) {
        this.plugins = plugins;
    }

    public void setMongodaoLocations(Resource[] mongodaoLocations) {
        this.mongodaoLocations = mongodaoLocations;
    }

    public MongoDaoResourceBuilderAdapter getMongoDaoResourceBuilderAdapter() {
        return mongoDaoResourceBuilderAdapter;
    }

    public void setMongoDaoResourceBuilderAdapter(MongoDaoResourceBuilderAdapter mongoDaoResourceBuilderAdapter) {
        this.mongoDaoResourceBuilderAdapter = mongoDaoResourceBuilderAdapter;
    }

    public void setMongoSessionFactory(MongoSessionFactory mongoSessionFactory) {
        this.mongoSessionFactory = mongoSessionFactory;
    }

    @Override
    public MongoSessionFactory getObject() throws Exception {
        if (this.mongoSessionFactory == null) {
            afterPropertiesSet();
        }
        return this.mongoSessionFactory;
    }

    @Override
    public Class<?> getObjectType() {
        return this.mongoSessionFactory == null ? MongoSessionFactory.class : this.mongoSessionFactory.getClass();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        notNull(mongoOperations, "Property 'mongoOperations' is required");
        this.mongoSessionFactory = buildMongoSessionFactory();
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    @Override
    public void onApplicationEvent(ApplicationEvent event) {
    }

    protected MongoSessionFactory buildMongoSessionFactory() {
        Configuration configuration;

        if (this.configuration != null) {
            configuration = this.configuration;
        } else {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Property 'configuration' not specified, using default Configuration");
            }
            configuration = new Configuration();
        }

        if (!ObjectUtils.isEmpty(this.plugins)) {
            for (Interceptor plugin : plugins) {
                configuration.addInterceptor(plugin);
                LOGGER.debug("Registered plugin: {}", plugin);
            }
        }

        if (resourceLoader == null) {
            resourceLoader = new PathMatchingResourcePatternResolver();
        }
        try {
            initMongoDaoStatement();
        } catch (Exception e) {
            LOGGER.error(e.toString());
        }
        return new DefaultMongoSessionFactory(configuration, mongoOperations);

    }

    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    protected void initMongoDaoStatement() {
        MetadataReaderFactory metaReader = new CachingMetadataReaderFactory(resourceLoader);
        if (mongodaoLocations != null) {
            for (Resource r : mongodaoLocations) {
                if (mongoDaoResourceBuilderAdapter != null) {
                    mongoDaoResourceBuilderAdapter.parse(r);
                }
            }
        }
    }

}
