package com.github.aidensuen.mongo.spring;

import com.github.aidensuen.mongo.session.ExecutorType;
import com.github.aidensuen.mongo.session.MongoSession;
import com.github.aidensuen.mongo.session.MongoSessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.TransientDataAccessResourceException;
import org.springframework.dao.support.PersistenceExceptionTranslator;
import org.springframework.transaction.support.ResourceHolderSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.Assert;

public final class MongoSessionUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(MongoSessionUtils.class);

    private MongoSessionUtils() {
    }

    public static MongoSession getMongoSession(MongoSessionFactory mongoSessionFactory) {
        ExecutorType executorType = mongoSessionFactory.getConfiguration().getDefaultExecutorType();
        return getMongoSession(mongoSessionFactory, executorType, null);
    }

    public static MongoSession getMongoSession(MongoSessionFactory mongoSessionFactory, ExecutorType executorType, PersistenceExceptionTranslator exceptionTranslator) {
        Assert.notNull(mongoSessionFactory, "no mongoSessionFactory specified");
        Assert.notNull(executorType, "no executorType specified");
        MongoSessionHolder holder = (MongoSessionHolder) TransactionSynchronizationManager.getResource(mongoSessionFactory);
        MongoSession session = sessionHolder(executorType, holder);
        if (holder != null) {
            return session;
        } else {
            LOGGER.debug("Creating a new MongoSession");
            session = mongoSessionFactory.getMongoSession(executorType);
            registerSessionHolder(mongoSessionFactory, executorType, exceptionTranslator, session);
            return session;
        }
    }

    private static void registerSessionHolder(MongoSessionFactory mongoSessionFactory, ExecutorType executorType, PersistenceExceptionTranslator exceptionTranslator, MongoSession session) {
        if (TransactionSynchronizationManager.isActualTransactionActive()) {
            LOGGER.debug("Registering transaction synchronization for mongoSession [{}]", session);
            MongoSessionHolder holder = new MongoSessionHolder(session, executorType, exceptionTranslator);
            TransactionSynchronizationManager.bindResource(mongoSessionFactory, holder);
            TransactionSynchronizationManager.registerSynchronization(new MongoSessionUtils.MongoSessionSynchronization(holder, mongoSessionFactory));
            holder.setSynchronizedWithTransaction(true);
            holder.requested();
        } else {
            LOGGER.debug("MongoSession [{}] was not register for synchronization because there is no transaction", session);
        }
    }

    private static MongoSession sessionHolder(ExecutorType executorType, MongoSessionHolder holder) {
        MongoSession session = null;
        if (holder != null && holder.isSynchronizedWithTransaction()) {
            if (holder.getExecutorType() != executorType) {
                throw new TransientDataAccessResourceException("Cannot change the ExecutorType when there is an existing transaction");
            }
            LOGGER.debug("Fetched MongoSession [{}] from current transaction", holder.getMongoSession());
            holder.requested();
            session = holder.getMongoSession();
        }
        return session;
    }

    private static final class MongoSessionSynchronization extends ResourceHolderSynchronization<MongoSessionHolder, MongoSessionFactory> {
        public MongoSessionSynchronization(MongoSessionHolder resourceHolder, MongoSessionFactory resourceKey) {
            super(resourceHolder, resourceKey);
        }
    }

}
