package com.github.aidensuen.mongo.spring.mongodao;

import com.github.aidensuen.mongo.common.Marker;
import com.github.aidensuen.mongo.core.MongoDaoRepository;
import com.github.aidensuen.mongo.session.MongoSessionFactory;
import com.github.aidensuen.mongo.spring.MongoSessionTemplate;
import org.springframework.beans.BeansException;
import org.springframework.beans.PropertyValue;
import org.springframework.beans.PropertyValues;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.PropertyResourceConfigurer;
import org.springframework.beans.factory.config.TypedStringValue;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.beans.factory.support.BeanNameGenerator;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.util.StringUtils;

import java.lang.annotation.Annotation;
import java.util.Map;
import java.util.Properties;

import static org.springframework.util.Assert.notNull;

public class MongoDaoScannerConfigurer implements BeanDefinitionRegistryPostProcessor, InitializingBean, ApplicationContextAware, BeanNameAware {

    private boolean addToConfig = true;

    private MongoSessionTemplate mongoSessionTemplate;

    private MongoSessionFactory mongoSessionFactory;

    private String mongoSessionFactoryBeanName;

    private String mongoSessionTemplateBeanName;

    private String basePackage;

    private Class<? extends Annotation> annotationClass;

    private Class<?> markerInterface;

    private String beanName;

    private BeanNameGenerator nameGenerator;

    private ApplicationContext applicationContext;

    private MongoDaoRepository mongoDaoRepository = new MongoDaoRepository();

    private boolean processPropertyPlaceHolders;

    public ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    @Override
    public void setBeanName(String name) {
        this.beanName = name;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        notNull(this.basePackage, "Property 'basePackage' is required");
    }

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        if (this.processPropertyPlaceHolders) {
            processPropertyPlaceHolders();
        }
        ClassPathMongoDaoScanner scanner = new ClassPathMongoDaoScanner(registry);
        scanner.setAnnotationClass(this.annotationClass);
        scanner.setMarkerInterface(this.markerInterface);
        scanner.setAddToConfig(this.addToConfig);
        scanner.setMongoSessionFactory(this.mongoSessionFactory);
        scanner.setMongoSessionFactoryBeanName(this.mongoSessionFactoryBeanName);
        scanner.setResourceLoader(this.applicationContext);
        scanner.setMongoSessionTemplate(this.mongoSessionTemplate);
        scanner.setMongoSessionTemplateBeanName(this.mongoSessionTemplateBeanName);
        scanner.setBeanNameGenerator(this.nameGenerator);
        scanner.registerFilters();

        //set generic dao
        scanner.setMongoDaoRepository(this.mongoDaoRepository);
        scanner.scan(StringUtils.tokenizeToStringArray(this.basePackage, ConfigurableApplicationContext.CONFIG_LOCATION_DELIMITERS));
    }

    private void processPropertyPlaceHolders() {
        Map<String, PropertyResourceConfigurer> prcs = applicationContext.getBeansOfType(PropertyResourceConfigurer.class);

        if (!prcs.isEmpty() && applicationContext instanceof ConfigurableApplicationContext) {
            BeanDefinition mapperScannerBean = ((ConfigurableApplicationContext) applicationContext)
                    .getBeanFactory().getBeanDefinition(beanName);

            // PropertyResourceConfigurer does not expose any methods to explicitly perform
            // property placeholder substitution. Instead, create a BeanFactory that just
            // contains this mapper scanner and post process the factory.
            DefaultListableBeanFactory factory = new DefaultListableBeanFactory();
            factory.registerBeanDefinition(beanName, mapperScannerBean);

            for (PropertyResourceConfigurer prc : prcs.values()) {
                prc.postProcessBeanFactory(factory);
            }

            PropertyValues values = mapperScannerBean.getPropertyValues();

            this.basePackage = updatePropertyValue("basePackage", values);
            this.mongoSessionFactoryBeanName = updatePropertyValue("mongoSessionFactoryBeanName", values);
            this.mongoSessionTemplateBeanName = updatePropertyValue("mongoSessionTemplateBeanName", values);
        }
    }


    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        // pass
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    public String getBasePackage() {
        return basePackage;
    }

    public void setBasePackage(String basePackage) {
        this.basePackage = basePackage;
    }

    public Class<? extends Annotation> getAnnotationClass() {
        return annotationClass;
    }

    public void setAnnotationClass(Class<? extends Annotation> annotationClass) {
        this.annotationClass = annotationClass;
    }

    public Class<?> getMarkerInterface() {
        return markerInterface;
    }

    public String getBeanName() {
        return beanName;
    }

    public void setAddToConfig(boolean addToConfig) {
        this.addToConfig = addToConfig;
    }

    public BeanNameGenerator getNameGenerator() {
        return nameGenerator;
    }

    public void setNameGenerator(BeanNameGenerator nameGenerator) {
        this.nameGenerator = nameGenerator;
    }

    public MongoSessionFactory getMongoSessionFactory() {
        return mongoSessionFactory;
    }

    public void setMongoSessionFactory(MongoSessionFactory mongoSessionFactory) {
        this.mongoSessionFactory = mongoSessionFactory;
    }

    public void setMongoSessionTemplate(MongoSessionTemplate mongoSessionTemplate) {
        this.mongoSessionTemplate = mongoSessionTemplate;
    }

    public MongoDaoRepository getMongoDaoRepository() {
        return mongoDaoRepository;
    }

    public void setMongoDaoRepository(MongoDaoRepository mongoDaoRepository) {
        this.mongoDaoRepository = mongoDaoRepository;
    }

    public void setProcessPropertyPlaceHolders(boolean processPropertyPlaceHolders) {
        this.processPropertyPlaceHolders = processPropertyPlaceHolders;
    }

    public String getMongoSessionFactoryBeanName() {
        return mongoSessionFactoryBeanName;
    }

    public void setMongoSessionFactoryBeanName(String mongoSessionFactoryBeanName) {
        this.mongoSessionFactoryBeanName = mongoSessionFactoryBeanName;
    }

    public void setProperties(Properties properties) {
        mongoDaoRepository.setProperties(properties);
    }

    public void setMarkerInterface(Class<?> markerInterface) {
        this.markerInterface = markerInterface;
        if (Marker.class.isAssignableFrom(markerInterface)) {
            mongoDaoRepository.registerMongoDao(markerInterface);
        }
    }

    private String updatePropertyValue(String propertyName, PropertyValues values) {
        PropertyValue property = values.getPropertyValue(propertyName);

        if (property == null) {
            return null;
        }

        Object value = property.getValue();

        if (value == null) {
            return null;
        } else if (value instanceof String) {
            return value.toString();
        } else if (value instanceof TypedStringValue) {
            return ((TypedStringValue) value).getValue();
        } else {
            return null;
        }
    }

    public void setMongoSessionTemplateBeanName(String mongoSessionTemplateBeanName) {
        this.mongoSessionTemplateBeanName = mongoSessionTemplateBeanName;
    }
}
