/*
 * Copyright 2017 David Karnok
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package hu.akarnokd.asyncenum;

import java.util.concurrent.*;
import java.util.function.*;

final class AsyncFromCompletionStage<T> implements AsyncEnumerable<T> {

    final CompletionStage<T> source;

    AsyncFromCompletionStage(CompletionStage<T> source) {
        this.source = source;
    }

    @Override
    public AsyncEnumerator<T> enumerator() {
        return new FromCompletionStageAsyncEnumerable<>(source);
    }

    static final class FromCompletionStageAsyncEnumerable<T>
            implements AsyncEnumerator<T>, BiConsumer<T, Throwable> {

        final CompletionStage<T> stage;

        CompletableFuture<Boolean> completable;

        boolean once;

        T result;

        FromCompletionStageAsyncEnumerable(CompletionStage<T> stage) {
            this.stage = stage;
        }

        @Override
        public CompletionStage<Boolean> moveNext() {
            if (once) {
                result = null;
                return FALSE;
            }
            once = true;
            CompletableFuture<Boolean> cf = new CompletableFuture<>();
            completable = cf;
            stage.whenComplete(this);
            return cf;
        }

        @Override
        public T current() {
            return result;
        }

        @Override
        public void accept(T t, Throwable throwable) {
            if (throwable != null) {
                completable.completeExceptionally(throwable);
            } else {
                result = t;
                completable.complete(true);
            }
        }

        @Override
        public void cancel() {
            // No action, consumer should stop calling moveNext().
        }
    }
}
