/*
 * FXGL - JavaFX Game Library. The MIT License (MIT).
 * Copyright (c) AlmasB (almaslvl@gmail.com).
 * See LICENSE for details.
 */

package com.almasb.fxgl.scene.intro

import com.almasb.fxgl.app.FXGL
import com.almasb.fxgl.scene.IntroScene
import javafx.animation.*
import javafx.geometry.Point2D
import javafx.geometry.Point3D
import javafx.scene.Group
import javafx.scene.canvas.Canvas
import javafx.scene.paint.Color
import javafx.scene.shape.Rectangle
import javafx.scene.text.Text
import javafx.util.Duration
import java.util.*

/**
 * This is the default FXGL Intro animation.
 *
 * @author Almas Baimagambetov (AlmasB) (almaslvl@gmail.com)
 */
class FXGLIntroScene() : IntroScene() {

    private val w: Double
    private val h: Double

    private val animation: ParallelTransition

    init {
        w = FXGL.getSettings().width.toDouble()
        h = FXGL.getSettings().height.toDouble()

        val f = makeLetter("F")
        val x = makeLetter("X")
        val g = makeLetter("G")
        val l = makeLetter("L")

        x.translateY = h + 70

        g.translateX = w

        l.translateX = w + 70
        l.translateY = h
        l.rotate = 180.0

        val fxglText = Group(f, x, g, l)

        val poweredText = makePoweredBy()
        val version = makeVersion()

        val content = Group(fxglText, poweredText, version)

        contentRoot.children.addAll(Rectangle(w, h), content)

        val originX = w / 2 - f.layoutBounds.width * 4 / 2
        val dx = f.layoutBounds.width

        val tt = TranslateTransition(Duration.seconds(1.0), f)
        tt.toX = originX
        tt.toY = h / 2

        val tt2 = TranslateTransition(Duration.seconds(1.0), x)
        tt2.toX = originX + dx
        tt2.toY = h / 2

        val tt3 = TranslateTransition(Duration.seconds(1.0), g)
        tt3.toX = originX + dx * 2
        tt3.toY = h / 2

        val tt4 = TranslateTransition(Duration.seconds(1.0), l)
        tt4.toX = originX + dx * 3.3
        tt4.toY = h / 2

        animation = ParallelTransition(tt, tt2, tt3, tt4)
        animation.setOnFinished { event ->
            poweredText.isVisible = true
            version.isVisible = true

            val rt = RotateTransition(Duration.seconds(1.0), l)
            rt.delay = Duration.seconds(0.66)
            rt.axis = Point3D(0.0, 0.0, 1.0)
            rt.byAngle = -180.0
            rt.setOnFinished { e ->
                animateParticles()
            }
            rt.play()
        }
    }

    private fun makeLetter(letter: String): Text {
        with(Text(letter)) {
            font = FXGL.getUIFactory().newFont(72.0)
            fill = Color.WHITESMOKE
            return this
        }
    }

    private fun makeVersion(): Text {
        with(Text("${com.almasb.fxgl.util.Version.getAsString()} by AlmasB")) {
            isVisible = false
            font = FXGL.getUIFactory().newFont(18.0)
            fill = Color.ALICEBLUE
            translateY = h - 5
            return this
        }
    }

    private fun makePoweredBy(): Text {
        with(Text("Powered By")) {
            isVisible = false
            font = FXGL.getUIFactory().newFont(18.0)
            fill = Color.WHITE
            translateX = (w - layoutBounds.width) / 2
            translateY = h / 2 - 80
            return this
        }
    }

    override fun startIntro() = animation.play()

    private fun animateParticles() {
        val particles = ArrayList<Particle>()

        val oldEffect = contentRoot.effect

        contentRoot.effect = null
        val image = contentRoot.snapshot(null, null)
        contentRoot.effect = oldEffect

        val reader = image.pixelReader
        for (y in 0..h.toInt() - 1) {
            for (x in 0..w.toInt() - 1) {
                val color = reader.getColor(x, y)

                if (color != Color.BLACK) {
                    particles.add(Particle(x.toDouble(), y.toDouble()))
                }
            }
        }

        val canvas = Canvas(w, h)
        val g = canvas.graphicsContext2D

        contentRoot.children.setAll(canvas)

        val timer = object : AnimationTimer() {
            override fun handle(now: Long) {

                if (particles.isEmpty()) {
                    stop()

                    val ft = FadeTransition(Duration.seconds(0.5), contentRoot)
                    ft.toValue = 0.0
                    ft.setOnFinished { e1 ->
                        finishIntro()
                    }
                    ft.play()
                }

                particles.removeIf { it.life == 0.0 }

                particles.filter({ p -> p.vel === Point2D.ZERO })
                        .sortedWith(Comparator { p1, p2 -> (p1.y - p2.y).toInt() })
                        .take(50)
                        .forEach { p -> p.vel = Point2D(Math.random() - 0.5, Math.random() - 0.5) }

                g.setGlobalAlpha(1.0)
                g.setFill(Color.BLACK)
                g.fillRect(0.0, 0.0, w, h)
                g.setFill(Color.WHITE)

                for (p in particles) {
                    p.update()

                    g.setGlobalAlpha(p.life)
                    g.fillOval(p.x, p.y, 1.0, 1.0)
                }
            }
        }
        timer.start()
    }

    private class Particle(var x: Double, var y: Double) {
        var vel = Point2D.ZERO
        var life = 1.0

        fun update() {
            life -= 0.017 * 0.45

            if (life < 0)
                life = 0.0

            x += vel.x
            y += vel.y
        }
    }
}