package amf.shapes.internal.domain.resolution.shape_normalization
import amf.core.client.scala.model.domain.Shape
import amf.core.internal.annotations.{AutoGeneratedName, TrackedElement}
import amf.core.internal.parser.domain.Annotations
import amf.shapes.client.scala.model.domain.{AnyShape, Example}
import amf.shapes.internal.domain.metamodel.AnyShapeModel
import scala.collection.mutable

object ExamplesCopier {
  def apply(shape: Shape, toShape: Shape): Unit = {
    (shape, toShape) match {
      case (from: AnyShape, to: AnyShape) =>
        copyExamples(from, to)
        if (to.examples.size > 1) checkExamplesNames(to.examples)
      case _ => // Nothing to do
    }
  }

  private def copyExamples(from: AnyShape, to: AnyShape): Unit = {
    from.examples.foreach(fromEx => {
      to.examples.find { toEx =>
        val sameId = fromEx.id == toEx.id
        sameId || areExamplesEqual(fromEx, toEx)
      } match {
        case Some(duplicatedExample) =>
          copyTracking(fromEx, duplicatedExample)
        case None =>
          to.setArrayWithoutId(AnyShapeModel.Examples, to.examples ++ Seq(fromEx))
      }
    })
  }

  /** 2 examples are equal when these 2 conditions are true:
    *   - their raw content is the same
    *   - their names are equal or not relevant (autogenerated or null)
    */
  private def areExamplesEqual(ex1: Example, ex2: Example): Boolean = {
    val oneIsGenerated = ex1.annotations.contains(classOf[AutoGeneratedName])
    val twoIsGenerated = ex2.annotations.contains(classOf[AutoGeneratedName])
    val sameRaw        = ex1.raw.option().getOrElse("").trim == ex2.raw.option().getOrElse("").trim
    val sameOrIrrelevantNames = {
      if (oneIsGenerated && twoIsGenerated) true
      else if (!oneIsGenerated && !twoIsGenerated) ex1.name.value() == ex2.name.value()
      else if (ex1.name.isNull || ex2.name.isNull) true // one name is generated, the other is null
      else false
    }
    sameRaw && sameOrIrrelevantNames
  }

  private def copyTracking(duplicate: Example, receiver: Example): Unit = {
    duplicate.annotations.find(classOf[TrackedElement]).foreach { dupAnnotation =>
      receiver.annotations += receiver.annotations
        .find(classOf[TrackedElement])
        .fold(TrackedElement(dupAnnotation.parents)) { receiverAnnotation =>
          receiver.annotations.reject(_.isInstanceOf[TrackedElement])
          TrackedElement(receiverAnnotation.parents ++ dupAnnotation.parents)
        }
    }
  }

  /** Generate a name for examples that have no name, or it's name is duplicated */
  private def checkExamplesNames(examples: Seq[Example]): Unit = {
    val definedNames = mutable.Set[String]()
    var index        = 0
    examples.foreach { example =>
      val exampleName = example.name.value()
      if (example.name.isNull || definedNames.contains(exampleName)) {
        val name = s"example_$index"
        definedNames.add(name)
        example.withName(name)
        example.add(Annotations(AutoGeneratedName()))
        index += 1
      } else definedNames.add(exampleName)
    }
  }
}
