package amf.shapes.client.scala.model.domain.jsonldinstance

import amf.core.client.platform.model.DataTypes
import amf.core.client.scala.model.domain.DomainElement
import amf.core.client.scala.vocabulary.ValueType
import amf.core.internal.metamodel.{Field, Obj, Type}
import amf.core.internal.parser.domain.{Annotations, Fields}
import amf.shapes.internal.domain.metamodel.jsonldschema.JsonLDEntityModel
import amf.shapes.internal.spec.jsonldschema.parser.JsonPath
import org.mulesoft.common.time.SimpleDateTime

object JsonLDObject {
  def empty(model: JsonLDEntityModel, path: JsonPath, annotations: Annotations = Annotations()): JsonLDObject =
    new JsonLDObject(Fields(), annotations, model, path)
}

case class JsonLDObject(
    override val fields: Fields,
    override val annotations: Annotations,
    private var model: JsonLDEntityModel,
    path: JsonPath
) extends DomainElement
    with JsonLDElement {
  override def meta: Obj = model

  /** Value , path + field value that is used to compose the id when the object its adopted */
  override def componentId = s"/${path.toString}"

  implicit class FieldBuilder(property: String) {

    def toObjField(meta: Obj): Field = Field(meta, ValueType(property))
    def toObjListField: Field        = Field(Type.Array(Type.ObjType), ValueType(property))

    def toStrField: Field     = Field(Type.Str, ValueType(property))
    def toStrListField: Field = Field(Type.Array(Type.Str), ValueType(property))

    def toIntField: Field     = Field(Type.Int, ValueType(property))
    def toIntListField: Field = Field(Type.Array(Type.Int), ValueType(property))

    def toBoolField: Field     = Field(Type.Bool, ValueType(property))
    def toBoolListField: Field = Field(Type.Array(Type.Bool), ValueType(property))

    def toFloatField: Field     = Field(Type.Float, ValueType(property))
    def toFloatListField: Field = Field(Type.Array(Type.Float), ValueType(property))

    def toDoubleField: Field     = Field(Type.Double, ValueType(property))
    def toDoubleListField: Field = Field(Type.Array(Type.Double), ValueType(property))

    def toDateField: Field     = Field(Type.Date, ValueType(property))
    def toDateListField: Field = Field(Type.Array(Type.Date), ValueType(property))

    def toDateTimeField: Field     = Field(Type.DateTime, ValueType(property))
    def toDateTimeListField: Field = Field(Type.Array(Type.DateTime), ValueType(property))
  }

  private def buildString(value: String) = new JsonLDScalar(value, DataTypes.String, annotations)

  private def buildDateTime(value: SimpleDateTime) = new JsonLDScalar(value, DataTypes.DateTime, annotations)
  private def buildDate(value: SimpleDateTime)     = new JsonLDScalar(value, DataTypes.Date, annotations)

  private def buildInteger(value: Int) = new JsonLDScalar(value, DataTypes.Integer, annotations)

  private def buildBoolean(value: Boolean) = new JsonLDScalar(value, DataTypes.Boolean, annotations)
  private def buildFloat(value: Float)     = new JsonLDScalar(value, DataTypes.Float, annotations)
  private def buildDouble(value: Double)   = new JsonLDScalar(value, DataTypes.Double, annotations)

  private def buildArray(values: Seq[JsonLDElement]) = JsonLDArray(values)

  private def updateModel(field: Field) = {
    if (model.fields.contains(field)) model // preserve initial order
    else model.copy(fields = model.fields :+ field)
  }
  private def updateModelAndSet(field: Field, element: JsonLDElement) = {
    model = updateModel(field)
    setWithoutId(field, element)
  }

  def withProperty(property: String, value: String): JsonLDObject =
    updateModelAndSet(property.toStrField, buildString(value))

  def withProperty(property: String, value: Int): JsonLDObject =
    updateModelAndSet(property.toIntField, buildInteger(value))

  def withProperty(property: String, value: Float): JsonLDObject =
    updateModelAndSet(property.toFloatField, buildFloat(value))

  def withProperty(property: String, value: Double): JsonLDObject =
    updateModelAndSet(property.toDoubleField, buildDouble(value))

  def withProperty(property: String, value: Boolean): JsonLDObject =
    updateModelAndSet(property.toBoolField, buildBoolean(value))

  def withProperty(property: String, value: JsonLDObject): JsonLDObject =
    updateModelAndSet(property.toObjField(value.meta), value)

  def withDateOnlyProperty(property: String, value: SimpleDateTime): JsonLDObject =
    updateModelAndSet(property.toDateField, buildDate(value))

  def withDateTimeProperty(property: String, value: SimpleDateTime): JsonLDObject =
    updateModelAndSet(property.toDateTimeField, buildDateTime(value))

  def withStringPropertyCollection(property: String, values: Seq[String]): JsonLDObject =
    updateModelAndSet(property.toStrListField, buildArray(values.map(buildString)))

  def withIntPropertyCollection(property: String, values: Seq[Int]): JsonLDObject =
    updateModelAndSet(property.toIntListField, buildArray(values.map(buildInteger)))

  def withFloatPropertyCollection(property: String, values: Seq[Float]): JsonLDObject =
    updateModelAndSet(property.toFloatListField, buildArray(values.map(buildFloat)))

  def withBoolPropertyCollection(property: String, values: Seq[Boolean]): JsonLDObject =
    updateModelAndSet(property.toBoolListField, buildArray(values.map(buildBoolean)))

  def withObjPropertyCollection(property: String, values: Seq[JsonLDObject]): JsonLDObject =
    updateModelAndSet(property.toObjListField, buildArray(values))

  def removeProperty(property: String): JsonLDObject = {
    // update model
    val newModelFields = model.fields.filterNot(f => f.value.iri() == property)
    model = model.copy(fields = newModelFields)

    // update fields
    fields.remove(property)

    this
  }

  def getModel: JsonLDEntityModel = model
}
