package amf.plugins.document.webapi.parser.spec.domain

import amf.core.annotations.SynthesizedField
import amf.core.model.domain.{AmfArray, AmfScalar}
import amf.core.parser.{Annotations, ScalarNode, _}
import amf.core.utils.AmfStrings
import amf.plugins.document.webapi.annotations.EmptyPayload
import amf.plugins.document.webapi.contexts.parser.raml.RamlWebApiContext
import amf.plugins.document.webapi.parser.spec.common.{AnnotationParser, SpecParserOps}
import amf.plugins.document.webapi.parser.spec.declaration.{DefaultType, AnyDefaultType}
import amf.plugins.document.webapi.vocabulary.VocabularyMappings
import amf.plugins.domain.shapes.models.ExampleTracking.tracking
import amf.plugins.domain.webapi.metamodel.{RequestModel, ResponseModel}
import amf.plugins.domain.webapi.models.{Parameter, Response, Payload}
import amf.validations.ParserSideValidations.UnsupportedExampleMediaTypeErrorSpecification
import org.yaml.model.{YType, YMap, YScalar, YMapEntry}

import scala.collection.mutable

/**
  *
  */
case class Raml10ResponseParser(entry: YMapEntry, adopt: Response => Unit, parseOptional: Boolean = false)(
    implicit ctx: RamlWebApiContext)
    extends RamlResponseParser(entry, adopt, parseOptional) {

  override def parseMap(response: Response, map: YMap): Unit = {
    AnnotationParser(response, map, List(VocabularyMappings.response)).parse()
  }

  override def supportsOptionalResponses: Boolean = false

  override protected val defaultType: DefaultType = AnyDefaultType
}

case class Raml08ResponseParser(entry: YMapEntry, adopt: Response => Unit, parseOptional: Boolean = false)(
    implicit ctx: RamlWebApiContext)
    extends RamlResponseParser(entry, adopt, parseOptional) {
  override protected def parseMap(response: Response, map: YMap): Unit = Unit

  override protected val defaultType: DefaultType = AnyDefaultType

  override def supportsOptionalResponses: Boolean = true
}

abstract class RamlResponseParser(entry: YMapEntry, adopt: Response => Unit, parseOptional: Boolean = false)(
    implicit ctx: RamlWebApiContext)
    extends SpecParserOps {

  protected def parseMap(response: Response, map: YMap)

  protected val defaultType: DefaultType

  def supportsOptionalResponses: Boolean

  def parse(): Response = {
    val node: AmfScalar = ScalarNode(entry.key).text()

    val response: Response = entry.value.tagType match {
      case YType.Null =>
        val response = Response(entry).set(ResponseModel.Name, node).set(ResponseModel.StatusCode, node)
        adopt(response)
        response
      case YType.Str | YType.Int =>
        val ref           = entry.value.as[YScalar].text
        val res: Response = ctx.declarations.findResponseOrError(entry.value)(ref, SearchScope.All).link(ref)
        res.set(ResponseModel.Name, node).annotations ++= Annotations(entry)
        res
      case _ =>
        val map = entry.value.as[YMap] // if not scalar, must be the response, if not, violation.
        val res = Response(entry).set(ResponseModel.Name, node)
        adopt(res)
        // res.withStatusCode(if (res.name.value() == "default") "200" else res.name.value())
        res.withStatusCode(res.name.value())

        if (parseOptional && node.toString.endsWith("?") && supportsOptionalResponses) {
          res.set(ResponseModel.Optional, value = true)
          val name = node.toString.stripSuffix("?")
          res.set(ResponseModel.Name, name)
          res.set(ResponseModel.StatusCode, name)
        }

        map.key("description", (ResponseModel.Description in res).allowingAnnotations)

        map.key("headers",
                (ResponseModel.Headers in res using RamlHeaderParser
                  .parse((p: Parameter) => p.adopted(res.id), parseOptional)).treatMapAsArray.optional)

        map.key(
          "body",
          entry => {
            val payloads = mutable.ListBuffer[Payload]()

            val payload = Payload(Annotations(entry))
            payload.adopted(res.id)

            entry.value.tagType match {
              case YType.Null =>
                ctx.factory
                  .typeParser(entry,
                              shape => tracking(shape.withName("default").adopted(payload.id), payload.id),
                              false,
                              defaultType)
                  .parse()
                  .foreach { schema =>
                    schema.annotations += SynthesizedField()
                    payload.annotations += EmptyPayload()
                    ParsingHelpers.autoGeneratedAnnotation(schema)
                    payloads += payload.withSchema(schema)
                  }
                res.set(RequestModel.Payloads, AmfArray(payloads, Annotations(entry.value)), Annotations(entry))

              case YType.Str =>
                ctx.factory
                  .typeParser(entry,
                              shape => tracking(shape.withName("default").adopted(payload.id), payload.id),
                              false,
                              defaultType)
                  .parse()
                  .foreach(s => {
                    ParsingHelpers.autoGeneratedAnnotation(s)
                    payloads += payload.withSchema(s)
                  })
                res.set(RequestModel.Payloads, AmfArray(payloads, Annotations(entry.value)), Annotations(entry))

              case _ =>
                // Now we parsed potentially nested shapes for different data types
                entry.value.to[YMap] match {
                  case Right(m) =>
                    val mediaTypeRegexPattern = ".*/.*"
                    m.regex(
                      mediaTypeRegexPattern,
                      entries => {
                        entries.foreach(entry => {
                          payloads += ctx.factory.payloadParser(entry, res.withPayload, false).parse()
                        })
                      }
                    )
                    val entries = m.entries.filter(e => !e.key.as[YScalar].text.matches(mediaTypeRegexPattern))
                    val others  = YMap(entries, entries.headOption.map(_.sourceName).getOrElse(""))
                    if (others.entries.nonEmpty) {
                      if (payloads.isEmpty) {
                        ctx.factory
                          .typeParser(entry,
                                      shape => shape.withName("default").adopted(payload.id),
                                      false,
                                      defaultType)
                          .parse()
                          .foreach { schema =>
                            val payload = res.withPayload()
                            ParsingHelpers.autoGeneratedAnnotation(schema)
                            payloads += payload
                              .add(Annotations(entry))
                              .withSchema(tracking(schema, payload.id))
                          }
                      } else {
                        others.entries.foreach(
                          e =>
                            ctx.eh.violation(
                              UnsupportedExampleMediaTypeErrorSpecification,
                              res.id,
                              s"Unexpected key '${e.key.as[YScalar].text}'. Expecting valid media types.",
                              e))
                      }
                    }
                  case _ =>
                }
                if (payloads.nonEmpty)
                  res.set(RequestModel.Payloads, AmfArray(payloads, Annotations(entry.value)), Annotations(entry))
            }
          }
        )

        map.key(
          "examples".asRamlAnnotation,
          entry => {
            val examples = OasResponseExamplesParser(entry).parse()
            res.set(ResponseModel.Examples, AmfArray(examples, Annotations(entry.value)), Annotations(entry))
          }
        )

        ctx.closedShape(res.id, map, "response")

        parseMap(res, map)
        res
    }

    response.annotations ++= Annotations(entry)
    response
  }
}
