package org.mulesoft.antlrast.ast

import scala.collection.mutable
import scala.scalajs.js.annotation.{JSExportAll, JSExportTopLevel}

@JSExportAll
trait ASTElement {
  val name: String
  val file: String
  val start: Position
  val end: Position

  def printAcc(indent: Int = 0, acc: StringBuffer = new StringBuffer()): StringBuffer = {
    this match {
      case node: Node =>
        acc.append(s"${" " * indent} - ${node.name} (${node.start.line},${node.start.column}) @ ${file}\n")
        for (elem <- node.children) {
          elem.printAcc(indent + 2, acc)
        }
        acc
      case elem: Terminal => acc.append(s"${" " * indent} => ${elem.value}\n")
      case error: Error   => acc.append(s"${" " * indent} !!! ${error.message}\n")
    }
  }

  def print(): Unit               = println(printAcc().toString)
  override def toString(): String = printAcc().toString
}

@JSExportTopLevel("Node")
@JSExportAll
case class Node(
    override val name: String,
    override val file: String,
    override val start: Position,
    override val end: Position,
    source: String,
    children: mutable.Buffer[ASTElement] = mutable.Buffer()
) extends ASTElement

object Node {
  def ROOT: Node = Node("", "", Position.Zero, Position.Zero, "")
}

@JSExportTopLevel("Terminal")
@JSExportAll
case class Terminal(
    override val name: String,
    override val file: String,
    override val start: Position,
    override val end: Position,
    value: String
) extends ASTElement

@JSExportTopLevel("Error")
@JSExportAll
case class Error(
    override val name: String,
    override val file: String,
    override val start: Position,
    override val end: Position,
    message: String
) extends ASTElement

@JSExportTopLevel("AST")
@JSExportAll
class AST(errors: mutable.Buffer[Error] = mutable.Buffer(), stack: mutable.Buffer[Node] = mutable.Buffer(Node.ROOT)) {

  def current(): Node = stack.last

  def push(node: Node): Unit = {
    current().children.append(node)
    stack.append(node)
  }

  def pop(): Unit = {
    stack.remove(stack.length - 1)
  }

  def error(error: Error): Unit = {
    current().children.append(error)
    errors.append(error)
  }

  def root(): ASTElement = stack.head.children.head

}

object AST {
  def apply() = new AST()
}
