package org.mulesoft.antlrast.ast

import org.antlr.v4.runtime.tree.{ErrorNode, ParseTreeListener, TerminalNode}
import org.mulesoft.common.client.lexical.SourceLocation
import org.antlr.v4.runtime.{ParserRuleContext, Token, Parser => ANTLRParser}

trait ASTBuilder extends ParseTreeListener {

  val file: String

  val parser: ANTLRParser

  var builder: AST = AST()

  def buildInputContext(): ParserRuleContext

  override def visitTerminal(terminal: TerminalNode): Unit = {
    val terminalType = terminal.getSymbol.getType
    var name         = parser.getVocabulary.getSymbolicName(terminalType)
    if (name == null) {
      name = parser.getVocabulary.getDisplayName(terminalType)
    }
    val token = terminal.getSymbol
    val node = Terminal(name, sourceLocation(token, token), terminal.getText)
    builder.current().children.append(node)
  }

  override def visitErrorNode(error: ErrorNode): Unit = {
    val token = error.getSymbol
    val err = Error("Error", sourceLocation(token, token), error.getText)
    builder.error(err)
  }

  override def enterEveryRule(ctx: ParserRuleContext): Unit = {
    val idx      = ctx.getRuleIndex
    val ruleName = parser.getRuleNames()(idx)
    val node = Node(
      ruleName,
      sourceLocation(ctx.start, ctx.stop),
      ctx.getText
    )
    builder.push(node)
  }

  override def exitEveryRule(ctx: ParserRuleContext): Unit = builder.pop()

  def ast(): AST = builder

  def sourceLocation(startToken: Token, stopToken: Token): SourceLocation = {
    val startLine   = startToken.getLine
    val startColumn = startToken.getCharPositionInLine
    val endLine     = stopToken.getLine
    val endColumn   = calculateEndColumn(stopToken)
    SourceLocation(file, startLine, startColumn, endLine, endColumn)
  }
  private def calculateEndColumn(token: Token) = {
    token.getCharPositionInLine + (token.getStopIndex - token.getStartIndex) + 1
  }
}
