package org.mulesoft.antlrast.unsafe

import org.mulesoft.common.io.{AsyncFile, FileSystem, SyncFile}

import scala.concurrent.{ExecutionContext, Future}

trait Platform {

  def name: String = "gen"

  /** Underlying file system for platform. */
  val fs: FileSystem

  /** Write specified content on given url. */
  def write(url: String, content: String)(implicit executionContext: ExecutionContext): Future[Unit] = {
    url match {
      case File(path) => writeFile(path, content)
      case _          => Future.failed(new Exception(s"Unsupported write operation: $url"))
    }
  }

  /** Return temporary directory. */
  def tmpdir(): String

  /** Return the OS (win, mac, nux). */
  def operativeSystem(): String

  /** Write specified content on specified file path. */
  protected def writeFile(path: String, content: String)(implicit executionContext: ExecutionContext): Future[Unit] =
    fs.asyncFile(path).write(content)

  protected def fixFilePrefix(res: String): String = {
    if (res.startsWith("file://") || res.startsWith("file:///")) {
      res
    } else if (res.startsWith("file:/")) {
      res.replace("file:/", "file:///")
    } else {
      res
    }
  }
}

object Platform {
  def base(url: String): Option[String] = Some(url.substring(0, url.lastIndexOf('/')))
}

object File {
  val FILE_PROTOCOL = "file://"

  def unapply(url: String): Option[String] = {
    url match {
      case s if s.startsWith(FILE_PROTOCOL) =>
        val path = s.stripPrefix(FILE_PROTOCOL)
        Some(path)
      case _ => None
    }
  }
}

/** Unsupported file system. */
object UnsupportedFileSystem extends FileSystem {

  override def syncFile(path: String): SyncFile = unsupported

  override def asyncFile(path: String): AsyncFile = unsupported

  override def separatorChar: Char = unsupported

  private def unsupported = throw new Exception(s"Unsupported operation")
}

case class FileNotFound(cause: Throwable) extends FileLoaderException("File Not Found: " + cause.getMessage, cause)

case class SocketTimeout(cause: Throwable) extends FileLoaderException("Socket Timeout: " + cause.getMessage, cause)

case class NetworkError(cause: Throwable) extends FileLoaderException("Network Error: " + cause.getMessage, cause)

case class UnexpectedStatusCode(resource: String, code: Int)
    extends Exception(s"Unexpected status code '$code' for resource '$resource'")

class UnsupportedUrlScheme(url: String) extends Exception("Unsupported Url scheme: " + url)

class PathResolutionError(message: String) extends Exception("Error resolving path: " + message)

abstract class FileLoaderException(msj: String, e: Throwable) extends Throwable(msj, e)
