package org.mulesoft.antlrast.errorHandler
import org.antlr.v4.runtime.{BaseErrorListener, CommonToken, RecognitionException, Recognizer, Token}
import org.mulesoft.antlrast.ast.Error
import org.mulesoft.antlrast.utils.SourceLocationBuilder
import org.mulesoft.common.client.lexical.SourceLocation
import scala.collection.mutable.Buffer

class GraphQLCustomErrorListener extends BaseErrorListener {
  var errors: Buffer[Error] = Buffer()

  override def syntaxError(
      recognizer: Recognizer[_, _],
      offendingSymbol: Any,
      line: Int,
      charPositionInLine: Int,
      msg: String,
      e: RecognitionException
  ): Unit = {
    val error = offendingSymbol match {
      case t: CommonToken if msg.startsWith("[AMF]") =>
        val kind     = "SyntaxError"
        val location = SourceLocationBuilder("", t, t)
        val message  = msg.stripPrefix("[AMF]")
        Error(kind, location, message)
      case token: CommonToken if isEndOfFileToken(token) => unexpectedEndOfFileError(token)
      case token: CommonToken                            => syntaxErrorAtParsing(token)
      case _                                             => lexerError(line, charPositionInLine, msg)
    }
    errors.append(error)
  }

  private def lexerError(line: Int, charPositionInLine: Int, msg: String) = {
    val unrecognizedToken = sanitizeMessage(msg.stripPrefix("token recognition error at: "))
    val message =
      if (unrecognizedToken != "") s"Syntax error in the following text: $unrecognizedToken"
      else "Syntax error:" ++ msg
    Error("SyntaxError", SourceLocation("", line, charPositionInLine), message)
  }

  private def syntaxErrorAtParsing(token: CommonToken) = {
    val message = s"Syntax error in the following text: '${token.getText}'"
    Error("SyntaxError", SourceLocationBuilder("", token, token), message)
  }

  private def unexpectedEndOfFileError(token: CommonToken) = {
    val message = s"Syntax error: Unexpected end of file"
    Error("SyntaxError", SourceLocationBuilder("", token, token), message)
  }

  private def isEndOfFileToken(token: CommonToken) = token.getText == "<EOF>"

  private def sanitizeMessage(msg: String): String = msg.replace("\n", "\\n")

}
