/*
 * Decompiled with CFR 0.152.
 */
package bt.tracker.http;

import bt.BtException;
import bt.metainfo.TorrentId;
import bt.net.Peer;
import bt.peer.IPeerRegistry;
import bt.protocol.crypto.EncryptionPolicy;
import bt.service.IdentityService;
import bt.torrent.TorrentDescriptor;
import bt.torrent.TorrentRegistry;
import bt.tracker.SecretKey;
import bt.tracker.Tracker;
import bt.tracker.TrackerRequestBuilder;
import bt.tracker.TrackerResponse;
import bt.tracker.http.CommonsHttpResponseHandler;
import bt.tracker.http.HttpResponseHandler;
import bt.tracker.http.urlencoding.TrackerQueryBuilder;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.InetAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.apache.http.HttpClientConnection;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.BasicHttpClientConnectionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpTracker
implements Tracker {
    private static final Logger LOGGER = LoggerFactory.getLogger(HttpTracker.class);
    private final URI baseUri;
    private final RequestConfig requestConfig;
    private final TorrentRegistry torrentRegistry;
    private final IdentityService idService;
    private final IPeerRegistry peerRegistry;
    private final EncryptionPolicy encryptionPolicy;
    private final int numberOfPeersToRequestFromTracker;
    private final CloseableHttpClient httpClient;
    private final CommonsHttpResponseHandler httpResponseHandler;
    private final ConcurrentMap<URI, byte[]> trackerIds;

    public HttpTracker(String trackerUrl, TorrentRegistry torrentRegistry, IdentityService idService, IPeerRegistry peerRegistry, EncryptionPolicy encryptionPolicy, InetAddress localAddress, int numberOfPeersToRequestFromTracker, Duration timeout) {
        try {
            this.baseUri = new URI(trackerUrl);
        }
        catch (URISyntaxException e) {
            throw new BtException("Invalid URL: " + trackerUrl, (Throwable)e);
        }
        this.torrentRegistry = torrentRegistry;
        this.idService = idService;
        this.peerRegistry = peerRegistry;
        this.encryptionPolicy = encryptionPolicy;
        this.numberOfPeersToRequestFromTracker = numberOfPeersToRequestFromTracker;
        this.requestConfig = this.buildReqConfig(localAddress, timeout);
        this.httpClient = HttpClients.createMinimal((HttpClientConnectionManager)new BasicHttpClientConnectionManager(){

            public synchronized void releaseConnection(HttpClientConnection conn, Object state, long keepalive, TimeUnit timeUnit) {
                try {
                    conn.close();
                }
                catch (IOException ex) {
                    LOGGER.debug("Error closing tracker connection.", (Throwable)ex);
                }
                super.releaseConnection(conn, state, keepalive, timeUnit);
            }
        });
        this.httpResponseHandler = new CommonsHttpResponseHandler(new HttpResponseHandler());
        this.trackerIds = new ConcurrentHashMap<URI, byte[]>();
    }

    private RequestConfig buildReqConfig(InetAddress localAddress, Duration timeout) {
        Duration trackerTimeout = timeout == null ? Duration.ofSeconds(30L) : timeout;
        return RequestConfig.custom().setLocalAddress(localAddress).setConnectTimeout((int)trackerTimeout.toMillis()).setSocketTimeout((int)trackerTimeout.toMillis()).build();
    }

    public TrackerRequestBuilder request(TorrentId torrentId) {
        return new TrackerRequestBuilder(torrentId){

            public TrackerResponse start() {
                return HttpTracker.this.sendEvent(TrackerRequestType.START, this);
            }

            public TrackerResponse stop() {
                return HttpTracker.this.sendEvent(TrackerRequestType.STOP, this);
            }

            public TrackerResponse complete() {
                return HttpTracker.this.sendEvent(TrackerRequestType.COMPLETE, this);
            }

            public TrackerResponse query() {
                return HttpTracker.this.sendEvent(TrackerRequestType.QUERY, this);
            }
        };
    }

    private TrackerResponse sendEvent(TrackerRequestType eventType, TrackerRequestBuilder requestBuilder) {
        String requestUri = this.buildQueryUri(eventType, requestBuilder);
        HttpGet request = new HttpGet(requestUri);
        request.setHeader("Connection", "close");
        request.setConfig(this.requestConfig);
        try {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Executing tracker HTTP request of type " + eventType.name() + "; request URL: " + requestUri);
            }
            return (TrackerResponse)this.httpClient.execute((HttpUriRequest)request, (ResponseHandler)this.httpResponseHandler);
        }
        catch (IOException e) {
            return TrackerResponse.exceptional((Throwable)e);
        }
    }

    private String buildQueryUri(TrackerRequestType eventType, TrackerRequestBuilder requestBuilder) {
        String requestUri;
        try {
            String query = this.buildQuery(eventType, requestBuilder);
            String baseUrl = this.baseUri.toASCIIString();
            if (baseUrl.endsWith("/")) {
                baseUrl = baseUrl.substring(0, baseUrl.length() - 1);
            }
            URL requestUrl = new URL(baseUrl + (this.baseUri.getRawQuery() == null ? "?" : "&") + query);
            requestUri = requestUrl.toURI().toString();
        }
        catch (Exception e) {
            throw new BtException("Failed to build tracker request", (Throwable)e);
        }
        return requestUri;
    }

    private String buildQuery(TrackerRequestType eventType, TrackerRequestBuilder requestBuilder) {
        TrackerQueryBuilder queryBuilder = this.createTrackerQuery(eventType, requestBuilder);
        return queryBuilder.toQuery();
    }

    protected TrackerQueryBuilder createTrackerQuery(TrackerRequestType eventType, TrackerRequestBuilder requestBuilder) {
        byte[] trackerId;
        TrackerQueryBuilder queryBuilder = new TrackerQueryBuilder();
        queryBuilder.add("info_hash", requestBuilder.getTorrentId().getBytes());
        queryBuilder.add("peer_id", this.idService.getLocalPeerId().getBytes());
        Peer peer = this.peerRegistry.getLocalPeer();
        InetAddress inetAddress = peer.getInetAddress();
        if (inetAddress != null) {
            queryBuilder.add("ip", inetAddress.getHostAddress());
        }
        queryBuilder.add("port", peer.getPort());
        this.torrentRegistry.getDescriptor(requestBuilder.getTorrentId()).flatMap(TorrentDescriptor::getSessionState).ifPresent(state -> {
            queryBuilder.add("uploaded", state.getUploaded());
            queryBuilder.add("downloaded", state.getDownloaded());
            long left = state.getLeft();
            if (left != -1L) {
                queryBuilder.add("left", state.getLeft());
            }
        });
        queryBuilder.add("compact", 1L);
        int numWant = requestBuilder.getNumWant() == null ? this.numberOfPeersToRequestFromTracker : requestBuilder.getNumWant();
        queryBuilder.add("numwant", numWant);
        Optional secretKey = this.idService.getSecretKey();
        if (secretKey.isPresent()) {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            ((SecretKey)secretKey.get()).writeTo((OutputStream)bos);
            queryBuilder.add("key", bos.toByteArray());
        }
        if ((trackerId = (byte[])this.trackerIds.get(this.baseUri)) != null) {
            queryBuilder.add("trackerid", trackerId);
        }
        if (null != eventType.getEventVal()) {
            queryBuilder.add("event", eventType.getEventVal());
        }
        switch (this.encryptionPolicy) {
            case PREFER_PLAINTEXT: 
            case PREFER_ENCRYPTED: {
                queryBuilder.add("supportcrypto", 1L);
                break;
            }
            case REQUIRE_ENCRYPTED: {
                queryBuilder.add("requirecrypto", 1L);
                break;
            }
        }
        return queryBuilder;
    }

    public String toString() {
        return "HttpTracker{baseUri=" + this.baseUri + '}';
    }

    public void close() {
        try {
            this.httpClient.close();
        }
        catch (IOException ex) {
            LOGGER.info("Error closing tracker http client", (Throwable)ex);
        }
    }

    protected static enum TrackerRequestType {
        START("started"),
        STOP("stopped"),
        COMPLETE("completed"),
        QUERY(null);

        private final String eventVal;

        private TrackerRequestType(String eventVal) {
            this.eventVal = eventVal;
        }

        public String getEventVal() {
            return this.eventVal;
        }
    }
}

