/*
 * Decompiled with CFR 0.152.
 */
package it.auties.curve25519;

import it.auties.curve25519.crypto.curve_sigs;
import it.auties.curve25519.crypto.scalarmult;
import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.interfaces.XECPrivateKey;
import java.security.interfaces.XECPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.NamedParameterSpec;
import java.security.spec.XECPrivateKeySpec;
import java.security.spec.XECPublicKeySpec;
import java.util.NoSuchElementException;
import java.util.Objects;

public class Curve25519 {
    private static final String KEY_ALGORITHM = "X25519";
    private static final int KEY_LENGTH = 32;
    private static final int SIGNATURE_LENGTH = 64;

    public static XECPrivateKey randomPrivateKey() {
        SecureRandom random = new SecureRandom();
        byte[] rawPrivateKey = new byte[32];
        random.nextBytes(rawPrivateKey);
        rawPrivateKey[0] = (byte)(rawPrivateKey[0] & 0xFFFFFFF8);
        rawPrivateKey[31] = (byte)(rawPrivateKey[31] & 0x7F);
        rawPrivateKey[31] = (byte)(rawPrivateKey[31] | 0x40);
        return Curve25519.createPrivateKey(rawPrivateKey);
    }

    public static KeyPair randomKeyPair() {
        XECPrivateKey privateKey = Curve25519.randomPrivateKey();
        XECPublicKey publicKey = Curve25519.getPublicKey(privateKey);
        return new KeyPair(publicKey, privateKey);
    }

    public static byte[] sharedKey(KeyPair keyPair) {
        Objects.requireNonNull(keyPair, "Key pair cannot be null!");
        return Curve25519.sharedKey(keyPair.getPublic(), keyPair.getPrivate());
    }

    public static byte[] sharedKey(PublicKey publicKey, PrivateKey privateKey) {
        Curve25519.checkPublicKeyType(publicKey);
        Curve25519.checkPrivateKeyType(privateKey);
        return Curve25519.sharedKey(Curve25519.readKey(publicKey), Curve25519.readKey(privateKey));
    }

    public static byte[] sharedKey(PublicKey publicKey, byte[] privateKey) {
        Curve25519.checkPublicKeyType(publicKey);
        return Curve25519.sharedKey(Curve25519.readKey(publicKey), privateKey);
    }

    public static byte[] sharedKey(byte[] publicKey, PrivateKey privateKey) {
        Curve25519.checkPrivateKeyType(privateKey);
        return Curve25519.sharedKey(publicKey, Curve25519.readKey(privateKey));
    }

    public static byte[] sharedKey(byte[] publicKey, byte[] privateKey) {
        Curve25519.checkKey(publicKey);
        Curve25519.checkKey(privateKey);
        byte[] agreement = new byte[32];
        scalarmult.crypto_scalarmult(agreement, privateKey, publicKey);
        return agreement;
    }

    public static byte[] sign(KeyPair keyPair, byte[] message, boolean deterministic) {
        return Curve25519.sign(keyPair, message, Curve25519.randomSignatureHash(deterministic));
    }

    public static byte[] sign(KeyPair keyPair, byte[] message, byte[] hash) {
        Objects.requireNonNull(keyPair, "Key pair cannot be null!");
        return Curve25519.sign(keyPair.getPrivate(), message, hash);
    }

    public static byte[] sign(PrivateKey privateKey, byte[] message, boolean deterministic) {
        return Curve25519.sign(privateKey, message, Curve25519.randomSignatureHash(deterministic));
    }

    public static byte[] sign(PrivateKey privateKey, byte[] message, byte[] hash) {
        Curve25519.checkPrivateKeyType(privateKey);
        return Curve25519.sign(Curve25519.readKey(privateKey), message, hash);
    }

    public static byte[] sign(byte[] privateKey, byte[] message, boolean deterministic) {
        return Curve25519.sign(privateKey, message, Curve25519.randomSignatureHash(deterministic));
    }

    public static byte[] sign(byte[] privateKey, byte[] message, byte[] hash) {
        Curve25519.checkKey(privateKey);
        Curve25519.checkHash(hash);
        byte[] signature = new byte[64];
        if (curve_sigs.curve25519_sign(signature, privateKey, message, message.length, hash) != 0) {
            throw new IllegalArgumentException("Message exceeds max length!");
        }
        return signature;
    }

    public static boolean verifySignature(PublicKey publicKey, byte[] message, byte[] signature) {
        Curve25519.checkPublicKeyType(publicKey);
        return Curve25519.verifySignature(Curve25519.readKey(publicKey), message, signature);
    }

    public static boolean verifySignature(byte[] publicKey, byte[] message, byte[] signature) {
        Curve25519.checkKey(publicKey);
        return message != null && signature != null && signature.length == 64 && curve_sigs.curve25519_verify(signature, publicKey, message, message.length) == 0;
    }

    public static XECPublicKey getPublicKey(PrivateKey privateKey) {
        Curve25519.checkPrivateKeyType(privateKey);
        return Curve25519.getPublicKey(Curve25519.readKey(privateKey));
    }

    public static XECPublicKey getPublicKey(byte[] privateKey) {
        byte[] rawPublicKey = new byte[32];
        curve_sigs.curve25519_keygen(rawPublicKey, privateKey);
        return Curve25519.createPublicKey(rawPublicKey);
    }

    public static XECPublicKey createPublicKey(byte[] rawPublicKey) {
        try {
            Objects.requireNonNull(rawPublicKey, "Public key cannot be null!");
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            XECPublicKeySpec xecPublicKeySpec = new XECPublicKeySpec(NamedParameterSpec.X25519, new BigInteger(Curve25519.convertKeyToJca(rawPublicKey)));
            return (XECPublicKey)keyFactory.generatePublic(xecPublicKeySpec);
        }
        catch (ClassCastException | NoSuchAlgorithmException exception) {
            throw new UnsupportedOperationException("Missing Curve25519 implementation", exception);
        }
        catch (InvalidKeySpecException exception) {
            throw new RuntimeException("Internal exception during key generation", exception);
        }
    }

    public static XECPrivateKey createPrivateKey(byte[] rawPrivateKey) {
        try {
            Objects.requireNonNull(rawPrivateKey, "Private key cannot be null!");
            KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            XECPrivateKeySpec xecPrivateKeySpec = new XECPrivateKeySpec(NamedParameterSpec.X25519, rawPrivateKey);
            return (XECPrivateKey)keyFactory.generatePrivate(xecPrivateKeySpec);
        }
        catch (ClassCastException | NoSuchAlgorithmException exception) {
            throw new UnsupportedOperationException("Missing Curve25519 implementation", exception);
        }
        catch (InvalidKeySpecException exception) {
            throw new RuntimeException("Internal exception during key generation", exception);
        }
    }

    public static byte[] readKey(PublicKey publicKey) {
        Objects.requireNonNull(publicKey, "Public key cannot be null!");
        Curve25519.checkPublicKeyType(publicKey);
        return Curve25519.convertKeyToJca(((XECPublicKey)publicKey).getU().toByteArray());
    }

    public static byte[] readKey(PrivateKey privateKey) {
        Objects.requireNonNull(privateKey, "Private key cannot be null!");
        Curve25519.checkPrivateKeyType(privateKey);
        return ((XECPrivateKey)privateKey).getScalar().orElseThrow(() -> new NoSuchElementException("Scalar content cannot be null!"));
    }

    private static byte[] randomSignatureHash(boolean deterministic) {
        if (deterministic) {
            return null;
        }
        byte[] random = new byte[64];
        new SecureRandom().nextBytes(random);
        return random;
    }

    private static void checkKey(byte[] key) {
        Objects.requireNonNull(key, "Key cannot be null!");
        if (key.length == 32) {
            return;
        }
        throw new IllegalArgumentException(String.format("Invalid key length: expected %s, got %s", 32, key.length));
    }

    private static void checkHash(byte[] hash) {
        if (hash == null || hash.length == 64) {
            return;
        }
        throw new IllegalArgumentException(String.format("Invalid hash length: expected %s, got %s", 64, hash.length));
    }

    private static void checkPublicKeyType(PublicKey publicKey) {
        if (!(publicKey instanceof XECPublicKey)) {
            throw new IllegalArgumentException("Invalid key type!");
        }
    }

    private static void checkPrivateKeyType(PrivateKey privateKey) {
        if (!(privateKey instanceof XECPrivateKey)) {
            throw new IllegalArgumentException("Invalid key type!");
        }
    }

    private static byte[] convertKeyToJca(byte[] arr) {
        byte[] result = new byte[32];
        int padding = result.length - arr.length;
        for (int i = 0; i < arr.length; ++i) {
            result[i + padding] = arr[arr.length - (i + 1)];
        }
        return result;
    }
}

