package com.bonnguyen.uberseekbar;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.SeekBar;
import android.widget.TextView;

import com.bonnguyen.uberseekbar.R;

/**
 * Created by nguyenbon on 3/28/16.
 */
public class UberSeekBar extends LinearLayout {

    private static final String TAG = "UberSeekBar";

    private boolean isShowTitle;
    private CharSequence[] titles;
    private Drawable[] bgItems;
    private Drawable thumbSeekBar;

    private int valueOfProgress = 0;
    private int lengthOfItems = 0;
    private int maxOfProgress = 0;

    private OnSeekBarChangeListener mOnSeekBarChangeListener;

    public UberSeekBar(Context context) {
        super(context);
    }

    public UberSeekBar(Context context, AttributeSet attrs) {
        super(context, attrs);
        TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.UberSeekBar, 0,
                0);
        try {
            isShowTitle = a.getBoolean(R.styleable.UberSeekBar_showTitle, false);
            maxOfProgress = a.getInteger(R.styleable.UberSeekBar_maxProgress, 0);
            if (maxOfProgress == 0) {
                throw new IllegalArgumentException(
                        "IntegerListPreference: error - max of progress is not zero");
            }
            valueOfProgress = a.getInteger(R.styleable.UberSeekBar_valueProgress, -1);
            if (valueOfProgress == -1) {
                throw new IllegalArgumentException(
                        "IntegerListPreference: error - max of progress is not less than zero");
            }
            thumbSeekBar = a.getDrawable(R.styleable.UberSeekBar_thumb);
            if (thumbSeekBar == null) {
                throw new IllegalArgumentException(
                        "IntegerListPreference: error - thumb is not null");
            }
            titles = a.getTextArray(R.styleable.UberSeekBar_titles);
            if (titles == null) {
                throw new IllegalArgumentException(
                        "IntegerListPreference: error - titles is not null");
            }
            int valuesResId = a.getResourceId(R.styleable.UberSeekBar_bgItems, -1);
            if (valuesResId == -1) {
                throw new IllegalArgumentException(
                        "IntegerListPreference: error - bg items is not zero");
            }

            TypedArray drawableArray = getResources().obtainTypedArray(valuesResId);
            lengthOfItems = drawableArray.length();
            if (lengthOfItems == 0) {
                throw new IllegalArgumentException(
                        "IntegerListPreference: error - items is not zero");
            }
            bgItems = new Drawable[lengthOfItems];
            for (int i = 0; i < lengthOfItems; i++) {
                bgItems[i] = drawableArray.getDrawable(i);
            }
            drawableArray.recycle();
        } finally {
            a.recycle();
        }

        LayoutInflater inflater = (LayoutInflater) context
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        View view = inflater.inflate(R.layout.layout_uber_seekbar, this, true);

        displayTitles(context, view);
        displayProgressDrawable(view);
    }

    public UberSeekBar(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
    }

    public void setOnSeekBarChangeListener(OnSeekBarChangeListener onSeekBarChangeListener) {
        this.mOnSeekBarChangeListener = onSeekBarChangeListener;
    }

    private void displayTitles(Context context, View view) {
        LinearLayout llTitle = (LinearLayout) view.findViewById(R.id.llTitle);
        if (isShowTitle && titles != null) {
            llTitle.setVisibility(View.VISIBLE);
            llTitle.removeAllViewsInLayout();
            for (CharSequence title : titles) {
                TextView tvTitle = new TextView(context);
                llTitle.addView(tvTitle);

                if (Build.VERSION.SDK_INT < 23) {
                    tvTitle.setTextAppearance(context, R.style.TitleTextViewStyle);
                } else {
                    tvTitle.setTextAppearance(R.style.TitleTextViewStyle);
                }
                tvTitle.setText(title);
                tvTitle.setGravity(Gravity.CENTER);
                tvTitle.setLayoutParams(new LinearLayout.LayoutParams(
                        ViewGroup.LayoutParams.MATCH_PARENT,
                        ViewGroup.LayoutParams.WRAP_CONTENT, 1));
            }
        } else {
            llTitle.setVisibility(View.GONE);
        }
    }

    private void displayProgressDrawable(View view) {
        SeekBar uberSeekBar = (SeekBar) view.findViewById(R.id.seekBar);
        uberSeekBar.setMax(maxOfProgress);
        uberSeekBar.setProgress(valueOfProgress);
        uberSeekBar.setThumb(thumbSeekBar);
        uberSeekBar.setProgressDrawable(new UberProgressDrawable(uberSeekBar.getProgressDrawable(), uberSeekBar, lengthOfItems, Color.GRAY));
        uberSeekBar.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(SeekBar seekBar, int value, boolean b) {
                if (mOnSeekBarChangeListener != null) {
                    mOnSeekBarChangeListener.onProgressChanged(seekBar, value, b);
                }
                valueOfProgress = value;
                if (valueOfProgress >= 0 && valueOfProgress <= 50) {
                    seekBar.setThumb(bgItems[0]);
                } else if (valueOfProgress > 50 && valueOfProgress < 150) {
                    seekBar.setThumb(bgItems[1]);
                } else if (valueOfProgress >= 150 && valueOfProgress < 250) {
                    seekBar.setThumb(bgItems[2]);
                } else if (valueOfProgress >= 250 && valueOfProgress <= 300) {
                    seekBar.setThumb(bgItems[3]);
                }
            }

            @Override
            public void onStartTrackingTouch(SeekBar seekBar) {
                if (mOnSeekBarChangeListener != null) {
                    mOnSeekBarChangeListener.onStartTrackingTouch(seekBar);
                }
            }

            @Override
            public void onStopTrackingTouch(SeekBar seekBar) {
                if (mOnSeekBarChangeListener != null) {
                    mOnSeekBarChangeListener.onStopTrackingTouch(seekBar);
                }
                if (valueOfProgress >= 0 && valueOfProgress <= 50) {
                    seekBar.setProgress(0);
                } else if (valueOfProgress > 50 && valueOfProgress < 150) {
                    seekBar.setProgress(100);
                } else if (valueOfProgress >= 150 && valueOfProgress < 250) {
                    seekBar.setProgress(200);
                } else if (valueOfProgress >= 250 && valueOfProgress <= 300) {
                    seekBar.setProgress(300);
                }
            }
        });
    }

    /**
     * A callback that notifies clients when the progress level has been
     * changed. This includes changes that were initiated by the user through a
     * touch gesture or arrow key/trackball as well as changes that were initiated
     * programmatically.
     */
    public interface OnSeekBarChangeListener {

        /**
         * Notification that the progress level has changed. Clients can use the fromUser parameter
         * to distinguish user-initiated changes from those that occurred programmatically.
         *
         * @param seekBar  The SeekBar whose progress has changed
         * @param progress The current progress level. This will be in the range 0..max where max
         *                 was set by {@link ProgressBar#setMax(int)}. (The default value for max is 100.)
         * @param fromUser True if the progress change was initiated by the user.
         */
        void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser);

        /**
         * Notification that the user has started a touch gesture. Clients may want to use this
         * to disable advancing the seekbar.
         *
         * @param seekBar The SeekBar in which the touch gesture began
         */
        void onStartTrackingTouch(SeekBar seekBar);

        /**
         * Notification that the user has finished a touch gesture. Clients may want to use this
         * to re-enable advancing the seekbar.
         *
         * @param seekBar The SeekBar in which the touch gesture began
         */
        void onStopTrackingTouch(SeekBar seekBar);
    }

}
