/*
 * Decompiled with CFR 0.152.
 */
package picard.sam;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileWriter;
import htsjdk.samtools.SAMFileWriterFactory;
import htsjdk.samtools.SAMProgramRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.CommandLineProgramProperties;
import picard.cmdline.Option;
import picard.cmdline.programgroups.SamOrBam;
import picard.sam.util.PhysicalLocation;

@CommandLineProgramProperties(usage="Class to downsample a BAM file while respecting that we should either get rid of both ends of a pair or neither \nend of the pair. In addition, this program uses the read-name and extracts the position within the tile whence \nthe read came from. The downsampling is based on this position. Results with the exact same input will produce the \nsame results.\n\nNote 1: This is technology and read-name dependent. If your read-names do not have coordinate information, or if your\nBAM contains reads from multiple technologies (flowcell versions, sequencing machines) this will not work properly. \nThis has been designed with Illumina MiSeq/HiSeq in mind.\nNote 2: The downsampling is not random. It is deterministically dependent on the position of the read within its tile.\nNote 3: Downsampling twice with this program is not supported.\nNote 4: You should call MarkDuplicates after downsampling.\n\nFinally, the code has been designed to simulate sequencing less as accurately as possible, not for getting an exact downsample \nfraction. In particular, since the reads may be distributed non-evenly within the lanes/tiles, the resulting downsampling \npercentage will not be accurately determined by the input argument FRACTION.", usageShort="Downsample a SAM or BAM file to retain a subset of the reads based on the reads location in each tile in the flowcell.", programGroup=SamOrBam.class)
public class PositionBasedDownsampleSam
extends CommandLineProgram {
    @Option(shortName="I", doc="The input SAM or BAM file to downsample.")
    public File INPUT;
    @Option(shortName="O", doc="The output, downsampled, SAM or BAM file to write.")
    public File OUTPUT;
    @Option(shortName="F", doc="The (approximate) fraction of reads to be kept, between 0 and 1.", optional=false)
    public Double FRACTION = null;
    @Option(doc="Stop after processing N reads, mainly for debugging.", optional=true)
    public Long STOP_AFTER = null;
    @Option(doc="Allow Downsampling again despite this being a bad idea with possibly unexpected results.", optional=true)
    public boolean ALLOW_MULTIPLE_DOWNSAMPLING_DESPITE_WARNINGS = false;
    @Option(doc="Determines whether the duplicate tag should be reset since the downsampling requires re-marking duplicates.")
    public boolean REMOVE_DUPLICATE_INFORMATION = true;
    private final Log log = Log.getInstance(PositionBasedDownsampleSam.class);
    private PhysicalLocation opticalDuplicateFinder;
    private long total = 0L;
    private long kept = 0L;
    public static String PG_PROGRAM_NAME = "PositionBasedDownsampleSam";
    private static final double ACCEPTABLE_FUDGE_FACTOR = 0.2;
    CollectionUtil.DefaultingMap.Factory<Coord, Short> defaultingMapFactory = new CollectionUtil.DefaultingMap.Factory<Coord, Short>(){

        public Coord make(Short s) {
            return new Coord();
        }
    };
    private final Map<Short, Coord> tileCoord = new CollectionUtil.DefaultingMap(this.defaultingMapFactory, true);
    final Map<Short, Histogram<Short>> xPositions = new HashMap<Short, Histogram<Short>>();
    final Map<Short, Histogram<Short>> yPositions = new HashMap<Short, Histogram<Short>>();

    public static void main(String[] stringArray) {
        new PositionBasedDownsampleSam().instanceMainWithExit(stringArray);
    }

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> arrayList = new ArrayList<String>();
        if (this.FRACTION < 0.0 || this.FRACTION > 1.0) {
            arrayList.add("FRACTION must be a value between 0 and 1, found: " + this.FRACTION);
        }
        if (arrayList.isEmpty()) {
            return null;
        }
        return arrayList.toArray(new String[arrayList.size()]);
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable((File)this.INPUT);
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        this.log.info(new Object[]{"Checking to see if input file has been downsampled with this program before."});
        this.checkProgramRecords();
        this.opticalDuplicateFinder = new PhysicalLocation();
        this.log.info(new Object[]{"Starting first pass. Examining read distribution in tiles."});
        this.fillTileMinMaxCoord();
        this.log.info(new Object[]{"First pass done."});
        this.log.info(new Object[]{"Starting second pass. Outputting reads."});
        this.outputSamRecords();
        this.log.info(new Object[]{"Second pass done."});
        double d = (double)this.kept / (double)this.total;
        if (Math.abs(d - this.FRACTION) / (Math.min(d, this.FRACTION) + 1.0E-10) > 0.2) {
            this.log.warn(new Object[]{String.format("You've requested FRACTION=%g, the resulting downsampling resulted in a rate of %f.", this.FRACTION, d)});
        }
        this.log.info(new Object[]{String.format("Finished! Kept %d out of %d reads (P=%g).", this.kept, this.total, d)});
        return 0;
    }

    private void outputSamRecords() {
        ProgressLogger progressLogger = new ProgressLogger(this.log, 10000000);
        SamReader samReader = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        SAMFileHeader sAMFileHeader = samReader.getFileHeader().clone();
        SAMFileHeader.PgIdGenerator pgIdGenerator = new SAMFileHeader.PgIdGenerator(sAMFileHeader);
        SAMProgramRecord sAMProgramRecord = new SAMProgramRecord(pgIdGenerator.getNonCollidingId(PG_PROGRAM_NAME));
        sAMProgramRecord.setProgramName(PG_PROGRAM_NAME);
        sAMProgramRecord.setCommandLine(this.getCommandLine());
        sAMProgramRecord.setProgramVersion(this.getVersion());
        sAMFileHeader.addProgramRecord(sAMProgramRecord);
        SAMFileWriter sAMFileWriter = new SAMFileWriterFactory().makeSAMOrBAMWriter(sAMFileHeader, true, this.OUTPUT);
        CircleSelector circleSelector = new CircleSelector(this.FRACTION);
        for (SAMRecord sAMRecord : samReader) {
            boolean bl;
            if (this.STOP_AFTER != null && this.total >= this.STOP_AFTER) break;
            ++this.total;
            PhysicalLocation physicalLocation = this.getSamRecordLocation(sAMRecord);
            if (!this.xPositions.containsKey(physicalLocation.getTile())) {
                this.xPositions.put(physicalLocation.getTile(), (Histogram<Short>)new Histogram(physicalLocation.getTile() + "-xpos", "count"));
            }
            if (!this.yPositions.containsKey(physicalLocation.getTile())) {
                this.yPositions.put(physicalLocation.getTile(), (Histogram<Short>)new Histogram(physicalLocation.getTile() + "-ypos", "count"));
            }
            if (bl = circleSelector.select(physicalLocation, this.tileCoord.get(physicalLocation.getTile()))) {
                if (this.REMOVE_DUPLICATE_INFORMATION) {
                    sAMRecord.setDuplicateReadFlag(false);
                }
                sAMFileWriter.addAlignment(sAMRecord);
                ++this.kept;
            }
            progressLogger.record(sAMRecord);
        }
        sAMFileWriter.close();
        CloserUtil.close((Object)samReader);
    }

    private void checkProgramRecords() {
        SamReader samReader = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        for (SAMProgramRecord sAMProgramRecord : samReader.getFileHeader().getProgramRecords()) {
            if (sAMProgramRecord.getProgramName() == null || !sAMProgramRecord.getProgramName().equals(PG_PROGRAM_NAME)) continue;
            String string = "Found previous Program Record that indicates that this BAM has been downsampled already with this program. Operation not supported! Previous PG: " + sAMProgramRecord.toString();
            if (this.ALLOW_MULTIPLE_DOWNSAMPLING_DESPITE_WARNINGS) {
                this.log.warn(new Object[]{string});
                continue;
            }
            this.log.error(new Object[]{string});
            throw new PicardException(string);
        }
        CloserUtil.close((Object)samReader);
    }

    private void fillTileMinMaxCoord() {
        SamReader samReader = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        ProgressLogger progressLogger = new ProgressLogger(this.log, 10000000, "Read");
        int n = 0;
        for (SAMRecord object : samReader) {
            if (this.STOP_AFTER != null && (long)n >= this.STOP_AFTER) break;
            ++n;
            progressLogger.record(object);
            PhysicalLocation physicalLocation = this.getSamRecordLocation(object);
            Coord coord = this.tileCoord.get(physicalLocation.getTile());
            coord.maxX = Math.max(coord.maxX, physicalLocation.getX());
            coord.minX = Math.min(coord.minX, physicalLocation.getX());
            coord.maxY = Math.max(coord.maxY, physicalLocation.getY());
            coord.minY = Math.min(coord.minY, physicalLocation.getY());
            ++coord.count;
        }
        for (Coord coord : this.tileCoord.values()) {
            int n2 = coord.maxX - coord.minX;
            int n3 = coord.maxY - coord.minY;
            coord.maxX += n2 / coord.count;
            coord.minX -= n2 / coord.count;
            coord.maxY += n3 / coord.count;
            coord.minY -= n3 / coord.count;
        }
        CloserUtil.close((Object)samReader);
    }

    private PhysicalLocation getSamRecordLocation(SAMRecord sAMRecord) {
        PhysicalLocation physicalLocation = new PhysicalLocation();
        this.opticalDuplicateFinder.addLocationInformation(sAMRecord.getReadName(), physicalLocation);
        return physicalLocation;
    }

    private class Coord {
        public int minX = 0;
        public int minY = 0;
        public int maxX = 0;
        public int maxY = 0;
        public int count = 0;
    }

    private class CircleSelector {
        private final double radiusSquared;
        private final double offset;
        private final boolean positiveSelection;

        CircleSelector(double d) {
            double d2;
            if (d > 0.5) {
                d2 = 1.0 - d;
                this.positiveSelection = false;
            } else {
                d2 = d;
                this.positiveSelection = true;
            }
            this.radiusSquared = d2 / Math.PI;
            if (d2 < 0.0) {
                throw new PicardException("This shouldn't happen...");
            }
            this.offset = Math.sqrt(this.radiusSquared - d2 * d2 / 4.0);
        }

        private double roundedPart(double d) {
            return d - (double)Math.round(d);
        }

        private boolean select(PhysicalLocation physicalLocation, Coord coord) {
            double d = Math.pow(this.roundedPart((double)(physicalLocation.getX() - coord.minX) / (double)(coord.maxX - coord.minX) - this.offset), 2.0) + Math.pow(this.roundedPart((double)(physicalLocation.getY() - coord.minY) / (double)(coord.maxY - coord.minY) - this.offset), 2.0);
            return d > this.radiusSquared ^ this.positiveSelection;
        }
    }
}

