/*
 * Decompiled with CFR 0.152.
 */
package picard.fingerprint;

import htsjdk.samtools.BamFileIoUtils;
import htsjdk.samtools.SAMReadGroupRecord;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.variant.utils.SAMSequenceDictionaryExtractor;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFHeader;
import java.io.File;
import java.util.Collections;
import java.util.List;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.analysis.FingerprintingDetailMetrics;
import picard.analysis.FingerprintingSummaryMetrics;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.Fingerprinting;
import picard.fingerprint.DiploidGenotype;
import picard.fingerprint.FingerprintChecker;
import picard.fingerprint.FingerprintResults;
import picard.fingerprint.LocusResult;
import picard.fingerprint.MatchResults;

@CommandLineProgramProperties(summary="Computes a fingerprint from the supplied input file (SAM/BAM or VCF) file and compares it to the expected fingerprint genotypes provided. The key output is a LOD score which represents the relative likelihood of the sequence data originating from the same sample as the genotypes vs. from a random sample.  Two outputs are produced: (1) a summary metrics file that gives metrics at the single sample level (if the input was a VCF) or at the read level (lane or index within a lane) (if the input was a SAM/BAM) versus a set of known genotypes for the expected sample, and (2) a detail metrics file that contains an individual SNP/Haplotype comparison within a fingerprint comparison.  The two files may be specified individually using the SUMMARY_OUTPUT and DETAIL_OUTPUT options.  Alternatively the OUTPUT option may be used instead to give the base of the two output files, with the summary metrics having a file extension 'fingerprinting_summary_metrics' and the detail metrics having a file extension 'fingerprinting_detail_metrics'.", oneLineSummary="Computes a fingerprint from the supplied input (SAM/BAM or VCF) file and compares it to the provided genotypes", programGroup=Fingerprinting.class)
@DocumentedFeature
public class CheckFingerprint
extends CommandLineProgram {
    static final String USAGE_DETAILS = "Computes a fingerprint from the supplied input file (SAM/BAM or VCF) file and compares it to the expected fingerprint genotypes provided. The key output is a LOD score which represents the relative likelihood of the sequence data originating from the same sample as the genotypes vs. from a random sample.  Two outputs are produced: (1) a summary metrics file that gives metrics at the single sample level (if the input was a VCF) or at the read level (lane or index within a lane) (if the input was a SAM/BAM) versus a set of known genotypes for the expected sample, and (2) a detail metrics file that contains an individual SNP/Haplotype comparison within a fingerprint comparison.  The two files may be specified individually using the SUMMARY_OUTPUT and DETAIL_OUTPUT options.  Alternatively the OUTPUT option may be used instead to give the base of the two output files, with the summary metrics having a file extension 'fingerprinting_summary_metrics' and the detail metrics having a file extension 'fingerprinting_detail_metrics'.";
    @Argument(shortName="I", doc="Input file SAM/BAM or VCF.  If a VCF is used, it must have at least one sample.  If there are more than one samples in the VCF, the parameter OBSERVED_SAMPLE_ALIAS must be provided in order to indicate which sample's data to use.  If there are no samples in the VCF, an exception will be thrown.")
    public File INPUT;
    @Argument(optional=true, doc="If the input is a VCF, this parameters used to select which sample's data in the VCF to use.")
    public String OBSERVED_SAMPLE_ALIAS;
    @Argument(shortName="O", doc="The base prefix of output files to write.  The summary metrics will have the file extension 'fingerprinting_summary_metrics' and the detail metrics will have the extension 'fingerprinting_detail_metrics'.", mutex={"SUMMARY_OUTPUT", "DETAIL_OUTPUT"})
    public String OUTPUT;
    @Argument(shortName="S", doc="The text file to which to write summary metrics.", mutex={"OUTPUT"})
    public File SUMMARY_OUTPUT;
    @Argument(shortName="D", doc="The text file to which to write detail metrics.", mutex={"OUTPUT"})
    public File DETAIL_OUTPUT;
    @Argument(shortName="G", doc="File of genotypes (VCF or GELI) to be used in comparison. May contain any number of genotypes; CheckFingerprint will use only those that are usable for fingerprinting.")
    public File GENOTYPES;
    @Argument(shortName="SAMPLE_ALIAS", optional=true, doc="This parameter can be used to specify which sample's genotypes to use from the expected VCF file (the GENOTYPES file).  If it is not supplied, the sample name from the input (VCF or BAM read group header) will be used.")
    public String EXPECTED_SAMPLE_ALIAS;
    @Argument(shortName="H", doc="The file lists a set of SNPs, optionally arranged in high-LD blocks, to be used for fingerprinting. See https://software.broadinstitute.org/gatk/documentation/article?id=9526 for details.")
    public File HAPLOTYPE_MAP;
    @Argument(shortName="LOD", doc="When counting haplotypes checked and matching, count only haplotypes where the most likely haplotype achieves at least this LOD.")
    public double GENOTYPE_LOD_THRESHOLD = 5.0;
    @Argument(optional=true, shortName="IGNORE_RG", doc="If the input is a SAM/BAM, and this parameter is true, treat the entire input BAM as one single read group in the calculation, ignoring RG annotations, and producing a single fingerprint metric for the entire BAM.")
    public boolean IGNORE_READ_GROUPS = false;
    private final Log log = Log.getInstance(CheckFingerprint.class);
    public static final String FINGERPRINT_SUMMARY_FILE_SUFFIX = "fingerprinting_summary_metrics";
    public static final String FINGERPRINT_DETAIL_FILE_SUFFIX = "fingerprinting_detail_metrics";

    public static void main(String[] args) {
        new CheckFingerprint().instanceMainWithExit(args);
    }

    @Override
    protected int doWork() {
        List<FingerprintResults> results;
        File outputSummaryMetricsFile;
        File outputDetailMetricsFile;
        if (this.OUTPUT == null) {
            outputDetailMetricsFile = this.DETAIL_OUTPUT;
            outputSummaryMetricsFile = this.SUMMARY_OUTPUT;
        } else {
            if (!this.OUTPUT.endsWith(".")) {
                this.OUTPUT = this.OUTPUT + ".";
            }
            outputDetailMetricsFile = new File(this.OUTPUT + FINGERPRINT_DETAIL_FILE_SUFFIX);
            outputSummaryMetricsFile = new File(this.OUTPUT + FINGERPRINT_SUMMARY_FILE_SUFFIX);
        }
        IOUtil.assertFileIsReadable((File)this.INPUT);
        IOUtil.assertFileIsReadable((File)this.HAPLOTYPE_MAP);
        IOUtil.assertFileIsReadable((File)this.GENOTYPES);
        IOUtil.assertFileIsWritable((File)outputDetailMetricsFile);
        IOUtil.assertFileIsWritable((File)outputSummaryMetricsFile);
        FingerprintChecker checker = new FingerprintChecker(this.HAPLOTYPE_MAP);
        String observedSampleAlias = null;
        boolean isBamOrSamFile = CheckFingerprint.isBamOrSamFile(this.INPUT);
        if (isBamOrSamFile) {
            SequenceUtil.assertSequenceDictionariesEqual((SAMSequenceDictionary)SAMSequenceDictionaryExtractor.extractDictionary((File)this.INPUT), (SAMSequenceDictionary)SAMSequenceDictionaryExtractor.extractDictionary((File)this.GENOTYPES), (boolean)true);
            SequenceUtil.assertSequenceDictionariesEqual((SAMSequenceDictionary)SAMSequenceDictionaryExtractor.extractDictionary((File)this.INPUT), (SAMSequenceDictionary)checker.getHeader().getSequenceDictionary(), (boolean)true);
            SamReader in = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
            for (SAMReadGroupRecord rec : in.getFileHeader().getReadGroups()) {
                if (observedSampleAlias == null) {
                    observedSampleAlias = rec.getSample();
                    continue;
                }
                if (observedSampleAlias.equals(rec.getSample())) continue;
                throw new PicardException("INPUT SAM/BAM file must not contain data from multiple samples.");
            }
            CloserUtil.close((Object)in);
            if (this.EXPECTED_SAMPLE_ALIAS == null) {
                this.EXPECTED_SAMPLE_ALIAS = observedSampleAlias;
            }
            results = checker.checkFingerprints(Collections.singletonList(this.INPUT), Collections.singletonList(this.GENOTYPES), this.EXPECTED_SAMPLE_ALIAS, this.IGNORE_READ_GROUPS);
        } else {
            VCFFileReader fileReader = new VCFFileReader(this.INPUT, false);
            VCFHeader fileHeader = fileReader.getFileHeader();
            if (fileHeader.getNGenotypeSamples() < 1) {
                throw new PicardException("INPUT VCF file must contain at least one sample.");
            }
            if (fileHeader.getNGenotypeSamples() > 1 && this.OBSERVED_SAMPLE_ALIAS == null) {
                throw new PicardException("INPUT VCF file contains multiple samples and yet the OBSERVED_SAMPLE_ALIAS parameter is not set.");
            }
            String string = observedSampleAlias = this.OBSERVED_SAMPLE_ALIAS != null ? this.OBSERVED_SAMPLE_ALIAS : (String)fileHeader.getGenotypeSamples().get(0);
            if (!fileHeader.getGenotypeSamples().contains(observedSampleAlias)) {
                throw new PicardException("INPUT VCF file does not contain OBSERVED_SAMPLE_ALIAS: " + observedSampleAlias);
            }
            if (this.OBSERVED_SAMPLE_ALIAS == null) {
                observedSampleAlias = (String)fileHeader.getGenotypeSamples().get(0);
            }
            fileReader.close();
            if (this.EXPECTED_SAMPLE_ALIAS == null) {
                this.EXPECTED_SAMPLE_ALIAS = observedSampleAlias;
            }
            results = checker.checkFingerprints(Collections.singletonList(this.INPUT), Collections.singletonList(this.GENOTYPES), observedSampleAlias, this.EXPECTED_SAMPLE_ALIAS);
        }
        MetricsFile summaryFile = this.getMetricsFile();
        MetricsFile detailsFile = this.getMetricsFile();
        for (FingerprintResults fpr : results) {
            MatchResults mr = fpr.getMatchResults().first();
            FingerprintingSummaryMetrics metrics = new FingerprintingSummaryMetrics();
            metrics.READ_GROUP = fpr.getReadGroup();
            metrics.SAMPLE = this.EXPECTED_SAMPLE_ALIAS;
            metrics.LL_EXPECTED_SAMPLE = mr.getSampleLikelihood();
            metrics.LL_RANDOM_SAMPLE = mr.getPopulationLikelihood();
            metrics.LOD_EXPECTED_SAMPLE = mr.getLOD();
            for (LocusResult lr : mr.getLocusResults()) {
                DiploidGenotype expectedGenotype = lr.getExpectedGenotype();
                DiploidGenotype observedGenotype = lr.getMostLikelyGenotype();
                ++metrics.HAPLOTYPES_WITH_GENOTYPES;
                if (lr.getLodGenotype() >= this.GENOTYPE_LOD_THRESHOLD) {
                    ++metrics.HAPLOTYPES_CONFIDENTLY_CHECKED;
                    if (lr.getExpectedGenotype() == lr.getMostLikelyGenotype()) {
                        ++metrics.HAPLOTYPES_CONFIDENTLY_MATCHING;
                    }
                    if (expectedGenotype.isHeterozygous() && observedGenotype.isHomomozygous()) {
                        ++metrics.HET_AS_HOM;
                    }
                    if (expectedGenotype.isHomomozygous() && observedGenotype.isHeterozygous()) {
                        ++metrics.HOM_AS_HET;
                    }
                    if (expectedGenotype.isHomomozygous() && observedGenotype.isHomomozygous() && expectedGenotype.compareTo(observedGenotype) != 0) {
                        ++metrics.HOM_AS_OTHER_HOM;
                    }
                }
                FingerprintingDetailMetrics details = new FingerprintingDetailMetrics();
                details.READ_GROUP = fpr.getReadGroup();
                details.SAMPLE = this.EXPECTED_SAMPLE_ALIAS;
                details.SNP = lr.getSnp().getName();
                details.SNP_ALLELES = lr.getSnp().getAlleleString();
                details.CHROM = lr.getSnp().getChrom();
                details.POSITION = lr.getSnp().getPos();
                details.EXPECTED_GENOTYPE = expectedGenotype.toString();
                details.OBSERVED_GENOTYPE = observedGenotype.toString();
                details.LOD = lr.getLodGenotype();
                details.OBS_A = lr.getAllele1Count();
                details.OBS_B = lr.getAllele2Count();
                detailsFile.addMetric((MetricBase)details);
            }
            summaryFile.addMetric((MetricBase)metrics);
            this.log.info(new Object[]{"Read Group: " + metrics.READ_GROUP + " / " + observedSampleAlias + " vs. " + metrics.SAMPLE + ": LOD = " + metrics.LOD_EXPECTED_SAMPLE});
        }
        summaryFile.write(outputSummaryMetricsFile);
        detailsFile.write(outputDetailMetricsFile);
        return 0;
    }

    @Override
    protected String[] customCommandLineValidation() {
        IOUtil.assertFileIsReadable((File)this.INPUT);
        boolean isBamOrSamFile = CheckFingerprint.isBamOrSamFile(this.INPUT);
        if (!isBamOrSamFile && this.IGNORE_READ_GROUPS) {
            return new String[]{"The parameter IGNORE_READ_GROUPS can only be used with BAM/SAM inputs."};
        }
        if (isBamOrSamFile && this.OBSERVED_SAMPLE_ALIAS != null) {
            return new String[]{"The parameter OBSERVED_SAMPLE_ALIAS can only be used with a VCF input."};
        }
        return super.customCommandLineValidation();
    }

    static boolean isBamOrSamFile(File f) {
        return BamFileIoUtils.isBamFile((File)f) || f.getName().endsWith(".sam");
    }
}

