/*
 * Decompiled with CFR 0.152.
 */
package picard.fingerprint;

import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.Fingerprinting;
import picard.fingerprint.CrosscheckMetric;
import picard.fingerprint.Fingerprint;
import picard.fingerprint.FingerprintChecker;
import picard.fingerprint.FingerprintIdDetails;
import picard.fingerprint.HaplotypeMap;
import picard.fingerprint.MatchResults;

@CommandLineProgramProperties(summary="Checks if all fingerprints within a set of files appear to come from the same individual. The fingerprints are calculated initially at the readgroup level (if present) but can be \"rolled-up\" by library, sample or file, to increase power and provide results at the desired resolution. Regular output is in a \"Moltenized\" format, one row per comparison. In this format the output will include the LOD score and also tumor-aware LOD score which can help assess identity even in the presence of a severe LOH sample with high purity. A matrix output is also available to facilitate visual inspection of crosscheck results.\nA separate CLP, ClusterCrosscheckMetrics, can cluster the results as a connected graph according to LOD greater than a threshold. ", oneLineSummary="Checks if all fingerprints appear to come from the same individual.", programGroup=Fingerprinting.class)
@DocumentedFeature
public class CrosscheckFingerprints
extends CommandLineProgram {
    @Argument(shortName="I", doc="One or more input files (or lists of files) to compare fingerprints for.")
    public List<File> INPUT;
    @Argument(shortName="SI", optional=true, mutex={"MATRIX_OUTPUT"}, doc="One or more input files (or lists of files) to compare fingerprints for. If this option is given the program compares each sample in INPUT with the sample from SECOND_INPUT that has the same sample ID. In addition, data will be grouped by SAMPLE regardless of the value of CROSSCHECK_BY. When operating in this mode, each sample in INPUT must also have a corresponding sample in SECOND_INPUT. If this is violated, the program will proceed to check the matching samples, but report the missing samples and return a non-zero error-code.")
    public List<File> SECOND_INPUT;
    @Argument(shortName="O", optional=true, doc="Optional output file to write metrics to. Default is to write to stdout.")
    public File OUTPUT = null;
    @Argument(shortName="MO", optional=true, doc="Optional output file to write matrix of LOD scores to. This is less informative than the metrics output and only contains Normal-Normal LOD score (i.e. doesn't account for Loss of heterogeneity). It is however sometimes easier to use visually.", mutex={"SECOND_INPUT"})
    public File MATRIX_OUTPUT = null;
    @Argument(shortName="H", doc="The file lists a set of SNPs, optionally arranged in high-LD blocks, to be used for fingerprinting. See https://software.broadinstitute.org/gatk/documentation/article?id=9526 for details.")
    public File HAPLOTYPE_MAP;
    @Argument(shortName="LOD", doc="If any two groups (with the same sample name) match with a LOD score lower than the threshold the program will exit with a non-zero code to indicate error. Program will also exit with an error if it finds two groups with different sample name that match with a LOD score greater than -LOD_THRESHOLD.\n\nLOD score 0 means equal likelihoodthat the groups match vs. come from different individuals, negative LOD scores mean N logs more likely that the groups are from different individuals, and positive numbers mean N logs more likely that the groups are from the sample individual. ")
    public double LOD_THRESHOLD = 0.0;
    @Argument(doc="Specificies which data-type should be used as the basic comparison unit. Fingerprints from readgroups can be \"rolled-up\" to the LIBRARY, SAMPLE, or FILE level before being compared. Fingerprints from VCF can be be compared by SAMPLE or FILE.")
    public CrosscheckMetric.DataType CROSSCHECK_BY = CrosscheckMetric.DataType.READGROUP;
    @Argument(doc="The number of threads to use to process files and generate Fingerprints.")
    public int NUM_THREADS = 1;
    @Argument(doc="Allow the use of duplicate reads in performing the comparison. Can be useful when duplicate marking has been overly aggressive and coverage is low.")
    public boolean ALLOW_DUPLICATE_READS = false;
    @Argument(doc="Assumed genotyping error rate that provides a floor on the probability that a genotype comes from the expected sample.")
    public double GENOTYPING_ERROR_RATE = 0.01;
    @Argument(doc="If true then only groups that do not relate to each other as expected will have their LODs reported.")
    public boolean OUTPUT_ERRORS_ONLY = false;
    @Argument(doc="The rate at which a heterozygous genotype in a normal sample turns into a homozygous (via loss of heterozygosity) in the tumor (model assumes independent events, so this needs to be larger than reality).", optional=true)
    public double LOSS_OF_HET_RATE = 0.5;
    @Argument(doc="Expect all groups' fingerprints to match, irrespective of their sample names.  By default (with this value set to false), groups (readgroups, libraries, files, or samples) with different sample names are expected to mismatch, and those with the same sample name are expected to match. ")
    public boolean EXPECT_ALL_GROUPS_TO_MATCH = false;
    @Argument(doc="When one or more mismatches between groups is detected, exit with this value instead of 0.")
    public int EXIT_CODE_WHEN_MISMATCH = 1;
    private final Log log = Log.getInstance(CrosscheckFingerprints.class);
    private double[][] crosscheckMatrix = null;
    private final List<String> matrixKeys = new ArrayList<String>();

    @Override
    protected int doWork() {
        int numUnexpected;
        IOUtil.assertFilesAreReadable(this.INPUT);
        IOUtil.assertFilesAreReadable(this.SECOND_INPUT);
        IOUtil.assertFileIsReadable((File)this.HAPLOTYPE_MAP);
        if (this.OUTPUT != null) {
            IOUtil.assertFileIsWritable((File)this.OUTPUT);
        }
        if (!this.SECOND_INPUT.isEmpty()) {
            this.log.info(new Object[]{"SECOND_INPUT is not empty. NOT doing cross-check. Will only compare each SAMPLE in INPUT against that sample in SECOND_INPUT."});
            if (this.CROSSCHECK_BY != CrosscheckMetric.DataType.SAMPLE) {
                this.log.warn(new Object[]{"CROSSCHECK_BY is not SAMPLE, This doesn't make sense in non-crosscheck mode. Setting CROSSCHECK_BY to SAMPLE."});
                this.CROSSCHECK_BY = CrosscheckMetric.DataType.SAMPLE;
            }
        }
        if (this.MATRIX_OUTPUT != null) {
            IOUtil.assertFileIsWritable((File)this.MATRIX_OUTPUT);
        }
        HaplotypeMap map = new HaplotypeMap(this.HAPLOTYPE_MAP);
        FingerprintChecker checker = new FingerprintChecker(map);
        checker.setAllowDuplicateReads(this.ALLOW_DUPLICATE_READS);
        checker.setValidationStringency(this.VALIDATION_STRINGENCY);
        ArrayList<String> extensions = new ArrayList<String>();
        extensions.add(".bam");
        extensions.add(".sam");
        extensions.addAll(Arrays.asList(IOUtil.VCF_EXTENSIONS));
        List unrolledFiles = IOUtil.unrollFiles(this.INPUT, (String[])extensions.toArray(new String[extensions.size()]));
        IOUtil.assertFilesAreReadable((List)unrolledFiles);
        List unrolledFiles2 = IOUtil.unrollFiles(this.SECOND_INPUT, (String[])extensions.toArray(new String[extensions.size()]));
        IOUtil.assertFilesAreReadable((List)unrolledFiles2);
        this.log.info(new Object[]{"Fingerprinting " + unrolledFiles.size() + " INPUT files."});
        Map<FingerprintIdDetails, Fingerprint> fpMap = checker.fingerprintFiles(unrolledFiles, this.NUM_THREADS, 1, TimeUnit.DAYS);
        ArrayList<CrosscheckMetric> metrics = new ArrayList<CrosscheckMetric>();
        if (this.SECOND_INPUT.isEmpty()) {
            this.log.info(new Object[]{"Cross-checking all " + (Object)((Object)this.CROSSCHECK_BY) + " against each other"});
            numUnexpected = this.crossCheckGrouped(fpMap, metrics, CrosscheckFingerprints.getFingerprintIdDetailsStringFunction(this.CROSSCHECK_BY), this.CROSSCHECK_BY);
        } else {
            this.log.info(new Object[]{"Fingerprinting " + unrolledFiles2.size() + " SECOND_INPUT files."});
            Map<FingerprintIdDetails, Fingerprint> fpMap2 = checker.fingerprintFiles(unrolledFiles2, this.NUM_THREADS, 1, TimeUnit.DAYS);
            this.log.info(new Object[]{"Checking each sample in INPUT with the same sample in SECOND_INPUT."});
            numUnexpected = this.checkFingerprintsBySample(fpMap, fpMap2, metrics);
        }
        MetricsFile metricsFile = this.getMetricsFile();
        metricsFile.addAllMetrics(metrics);
        if (this.OUTPUT != null) {
            metricsFile.write(this.OUTPUT);
        } else {
            metricsFile.write((Writer)new OutputStreamWriter(System.out));
        }
        if (this.MATRIX_OUTPUT != null) {
            this.writeMatrix();
        }
        if (numUnexpected > 0) {
            this.log.warn(new Object[]{"At least two read groups did not relate as expected."});
            return this.EXIT_CODE_WHEN_MISMATCH;
        }
        this.log.info(new Object[]{"All read groups related as expected."});
        return 0;
    }

    private void writeMatrix() {
        NumberFormat format = NumberFormat.getInstance();
        format.setMaximumFractionDigits(4);
        try (FileOutputStream stream = new FileOutputStream(this.MATRIX_OUTPUT);
             BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(stream));){
            int i;
            writer.write(this.CROSSCHECK_BY.name());
            for (i = 0; i < this.crosscheckMatrix.length; ++i) {
                writer.write('\t' + this.matrixKeys.get(i));
            }
            writer.newLine();
            for (i = 0; i < this.crosscheckMatrix.length; ++i) {
                writer.write(this.matrixKeys.get(i));
                for (int j = 0; j < this.crosscheckMatrix.length; ++j) {
                    writer.write('\t' + format.format(this.crosscheckMatrix[i][j]));
                }
                writer.newLine();
            }
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static Function<FingerprintIdDetails, String> getFingerprintIdDetailsStringFunction(CrosscheckMetric.DataType CROSSCHECK_BY) {
        Function<FingerprintIdDetails, String> groupByTemp;
        switch (CROSSCHECK_BY) {
            case READGROUP: {
                groupByTemp = details -> details.platformUnit;
                break;
            }
            case LIBRARY: {
                groupByTemp = details -> details.sample + "::" + details.library;
                break;
            }
            case FILE: {
                groupByTemp = details -> details.file + "::" + details.sample;
                break;
            }
            case SAMPLE: {
                groupByTemp = details -> details.sample;
                break;
            }
            default: {
                throw new PicardException("unpossible");
            }
        }
        return key -> {
            String temp = (String)groupByTemp.apply((FingerprintIdDetails)key);
            return temp == null ? Integer.toString(key.hashCode()) : temp;
        };
    }

    public static Map<FingerprintIdDetails, Fingerprint> mergeFingerprintsBy(Map<FingerprintIdDetails, Fingerprint> fingerprints, Function<FingerprintIdDetails, String> by) {
        Map<String, List<Map.Entry>> collection = fingerprints.entrySet().stream().collect(Collectors.groupingBy(entry -> (String)by.apply((FingerprintIdDetails)entry.getKey())));
        return collection.entrySet().stream().collect(Collectors.toMap(entry -> {
            FingerprintIdDetails finalId = new FingerprintIdDetails();
            ((List)entry.getValue()).forEach(id -> finalId.merge((FingerprintIdDetails)id.getKey()));
            finalId.group = (String)entry.getKey();
            return finalId;
        }, entry -> {
            FingerprintIdDetails firstDetail = (FingerprintIdDetails)((Map.Entry)((List)entry.getValue()).get(0)).getKey();
            Fingerprint sampleFp = new Fingerprint(firstDetail.sample, null, (String)by.apply(firstDetail));
            ((List)entry.getValue()).stream().map(Map.Entry::getValue).collect(Collectors.toSet()).forEach(sampleFp::merge);
            return sampleFp;
        }));
    }

    private int crossCheckGrouped(Map<FingerprintIdDetails, Fingerprint> fingerprints, List<CrosscheckMetric> metrics, Function<FingerprintIdDetails, String> by, CrosscheckMetric.DataType type) {
        Map<FingerprintIdDetails, Fingerprint> fingerprintsByGroup = CrosscheckFingerprints.mergeFingerprintsBy(fingerprints, by);
        if (this.MATRIX_OUTPUT != null) {
            this.crosscheckMatrix = new double[fingerprintsByGroup.size()][];
            for (int i = 0; i < fingerprintsByGroup.size(); ++i) {
                this.crosscheckMatrix[i] = new double[fingerprintsByGroup.size()];
            }
        }
        return this.crossCheckFingerprints(fingerprintsByGroup, type, metrics, by);
    }

    private int crossCheckFingerprints(Map<FingerprintIdDetails, Fingerprint> fingerprints, CrosscheckMetric.DataType type, List<CrosscheckMetric> metrics, Function<FingerprintIdDetails, String> by) {
        int unexpectedResults = 0;
        long checksMade = 0L;
        int logEvery = 100000;
        ArrayList<FingerprintIdDetails> fingerprintIdDetails = new ArrayList<FingerprintIdDetails>(fingerprints.keySet());
        long totalChecks = (long)fingerprintIdDetails.size() * ((long)fingerprintIdDetails.size() + 1L) / 2L;
        for (int i = 0; i < fingerprintIdDetails.size(); ++i) {
            FingerprintIdDetails lhsRg = (FingerprintIdDetails)fingerprintIdDetails.get(i);
            if (this.MATRIX_OUTPUT != null) {
                this.matrixKeys.add(by.apply(lhsRg));
            }
            for (int j = i; j < fingerprintIdDetails.size(); ++j) {
                FingerprintIdDetails rhsRg = (FingerprintIdDetails)fingerprintIdDetails.get(j);
                boolean expectedToMatch = this.EXPECT_ALL_GROUPS_TO_MATCH || lhsRg.sample.equals(rhsRg.sample);
                MatchResults results = FingerprintChecker.calculateMatchResults(fingerprints.get(lhsRg), fingerprints.get(rhsRg), this.GENOTYPING_ERROR_RATE, this.LOSS_OF_HET_RATE);
                CrosscheckMetric.FingerprintResult result = this.getMatchResults(expectedToMatch, results);
                if (!this.OUTPUT_ERRORS_ONLY || result == CrosscheckMetric.FingerprintResult.INCONCLUSIVE || !result.isExpected().booleanValue()) {
                    metrics.add(this.getMatchDetails(result, results, lhsRg, rhsRg, type));
                }
                if (result != CrosscheckMetric.FingerprintResult.INCONCLUSIVE && !result.isExpected().booleanValue()) {
                    ++unexpectedResults;
                }
                if (this.crosscheckMatrix != null) {
                    this.crosscheckMatrix[i][j] = results.getLOD();
                    this.crosscheckMatrix[j][i] = results.getLOD();
                }
                if (++checksMade % 100000L != 0L) continue;
                this.log.info(new Object[]{"Compared " + checksMade + " of " + totalChecks});
            }
        }
        return unexpectedResults;
    }

    private int checkFingerprintsBySample(Map<FingerprintIdDetails, Fingerprint> fingerprints1, Map<FingerprintIdDetails, Fingerprint> fingerprints2, List<CrosscheckMetric> metrics) {
        int unexpectedResults = 0;
        Map<FingerprintIdDetails, Fingerprint> fingerprints1BySample = CrosscheckFingerprints.mergeFingerprintsBy(fingerprints1, CrosscheckFingerprints.getFingerprintIdDetailsStringFunction(CrosscheckMetric.DataType.SAMPLE));
        Map<FingerprintIdDetails, Fingerprint> fingerprints2BySample = CrosscheckFingerprints.mergeFingerprintsBy(fingerprints2, CrosscheckFingerprints.getFingerprintIdDetailsStringFunction(CrosscheckMetric.DataType.SAMPLE));
        Map<String, FingerprintIdDetails> sampleToDetail1 = fingerprints1BySample.keySet().stream().collect(Collectors.toMap(id -> id.group, id -> id));
        Map<String, FingerprintIdDetails> sampleToDetail2 = fingerprints2BySample.keySet().stream().collect(Collectors.toMap(id -> id.group, id -> id));
        HashSet<String> samples = new HashSet<String>();
        samples.addAll(sampleToDetail1.keySet());
        samples.addAll(sampleToDetail2.keySet());
        for (String sample : samples) {
            FingerprintIdDetails lhsID = sampleToDetail1.get(sample);
            FingerprintIdDetails rhsID = sampleToDetail2.get(sample);
            if (lhsID == null || rhsID == null) {
                this.log.error(new Object[]{String.format("sample %s is missing from %s group", sample, lhsID == null ? "LEFT" : "RIGHT")});
                ++unexpectedResults;
                continue;
            }
            MatchResults results = FingerprintChecker.calculateMatchResults(fingerprints1BySample.get(lhsID), fingerprints2BySample.get(rhsID), this.GENOTYPING_ERROR_RATE, this.LOSS_OF_HET_RATE);
            CrosscheckMetric.FingerprintResult result = this.getMatchResults(true, results);
            if (!this.OUTPUT_ERRORS_ONLY || !result.isExpected().booleanValue()) {
                metrics.add(this.getMatchDetails(result, results, lhsID, rhsID, CrosscheckMetric.DataType.SAMPLE));
            }
            if (result == CrosscheckMetric.FingerprintResult.INCONCLUSIVE || result.isExpected().booleanValue()) continue;
            ++unexpectedResults;
        }
        return unexpectedResults;
    }

    private CrosscheckMetric getMatchDetails(CrosscheckMetric.FingerprintResult matchResult, MatchResults results, FingerprintIdDetails leftPuDetails, FingerprintIdDetails rightPuDetails, CrosscheckMetric.DataType type) {
        CrosscheckMetric metric = new CrosscheckMetric();
        metric.LEFT_GROUP_VALUE = leftPuDetails.group;
        metric.RIGHT_GROUP_VALUE = rightPuDetails.group;
        metric.RESULT = matchResult;
        metric.LOD_SCORE = results.getLOD();
        metric.LOD_SCORE_TUMOR_NORMAL = results.getLodTN();
        metric.LOD_SCORE_NORMAL_TUMOR = results.getLodNT();
        metric.DATA_TYPE = type;
        metric.LEFT_RUN_BARCODE = leftPuDetails.runBarcode;
        metric.LEFT_LANE = leftPuDetails.runLane;
        metric.LEFT_MOLECULAR_BARCODE_SEQUENCE = leftPuDetails.molecularBarcode;
        metric.LEFT_LIBRARY = leftPuDetails.library;
        metric.LEFT_SAMPLE = leftPuDetails.sample;
        metric.LEFT_FILE = leftPuDetails.file;
        metric.RIGHT_RUN_BARCODE = rightPuDetails.runBarcode;
        metric.RIGHT_LANE = rightPuDetails.runLane;
        metric.RIGHT_MOLECULAR_BARCODE_SEQUENCE = rightPuDetails.molecularBarcode;
        metric.RIGHT_LIBRARY = rightPuDetails.library;
        metric.RIGHT_SAMPLE = rightPuDetails.sample;
        metric.RIGHT_FILE = rightPuDetails.file;
        return metric;
    }

    private CrosscheckMetric.FingerprintResult getMatchResults(boolean expectedToMatch, MatchResults results) {
        if (expectedToMatch) {
            if (results.getLOD() < this.LOD_THRESHOLD) {
                return CrosscheckMetric.FingerprintResult.UNEXPECTED_MISMATCH;
            }
            if (results.getLOD() > -this.LOD_THRESHOLD) {
                return CrosscheckMetric.FingerprintResult.EXPECTED_MATCH;
            }
            return CrosscheckMetric.FingerprintResult.INCONCLUSIVE;
        }
        if (results.getLOD() > -this.LOD_THRESHOLD) {
            return CrosscheckMetric.FingerprintResult.UNEXPECTED_MATCH;
        }
        if (results.getLOD() < this.LOD_THRESHOLD) {
            return CrosscheckMetric.FingerprintResult.EXPECTED_MISMATCH;
        }
        return CrosscheckMetric.FingerprintResult.INCONCLUSIVE;
    }
}

