/*
 * Decompiled with CFR 0.152.
 */
package picard.util;

import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMProgramRecord;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Interval;
import htsjdk.samtools.util.IntervalList;
import htsjdk.samtools.util.Log;
import htsjdk.variant.vcf.VCFFileReader;
import java.io.File;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineParser;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.Intervals;
import picard.util.IntervalListScatterer;

@CommandLineProgramProperties(summary="Manipulates interval lists.  This tool offers multiple interval list file manipulation capabilities include sorting, merging, subtracting, padding, customizing, and other set-theoretic operations. If given one or more inputs, the default operation is to merge and sort them.  Other options e.g. interval subtraction are controlled by the arguments. The tool lists intervals with respect to a reference sequence.<br /><br />Both interval_list and VCF files are accepted as input. The interval_list file format is relatively simple and reflects the SAM alignment format to a degree.  A SAM style header must be present in the file that lists the sequence records against which the intervals are described.  After the header, the file then contains records, one per line in text format with the following values tab-separated: <pre>     -Sequence name (SN) <br />     -Start position (1-based)** <br />     -End position (1-based, end inclusive) <br />     -Strand (either + or -) <br />     -Interval name (ideally unique names for intervals)</pre>The coordinate system of interval_list files is such that the first base or position in a sequence is position \"1\".<h4>Usage example:</h4><pre>java -jar picard.jar IntervalListTools \\<br />      I=input.interval_list \\<br />      SI=input_2.interval_list \\<br />      O=new.interval_list</pre><hr />", oneLineSummary="Manipulates interval lists.  ", programGroup=Intervals.class)
@DocumentedFeature
public class IntervalListTools
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Manipulates interval lists.  ";
    static final String USAGE_DETAILS = "This tool offers multiple interval list file manipulation capabilities include sorting, merging, subtracting, padding, customizing, and other set-theoretic operations. If given one or more inputs, the default operation is to merge and sort them.  Other options e.g. interval subtraction are controlled by the arguments. The tool lists intervals with respect to a reference sequence.<br /><br />Both interval_list and VCF files are accepted as input. The interval_list file format is relatively simple and reflects the SAM alignment format to a degree.  A SAM style header must be present in the file that lists the sequence records against which the intervals are described.  After the header, the file then contains records, one per line in text format with the following values tab-separated: <pre>     -Sequence name (SN) <br />     -Start position (1-based)** <br />     -End position (1-based, end inclusive) <br />     -Strand (either + or -) <br />     -Interval name (ideally unique names for intervals)</pre>The coordinate system of interval_list files is such that the first base or position in a sequence is position \"1\".<h4>Usage example:</h4><pre>java -jar picard.jar IntervalListTools \\<br />      I=input.interval_list \\<br />      SI=input_2.interval_list \\<br />      O=new.interval_list</pre><hr />";
    @Argument(shortName="I", doc="One or more interval lists. If multiple interval lists are provided the output is theresult of merging the inputs. Supported formats are interval_list and VCF.", minElements=1)
    public List<File> INPUT;
    @Argument(doc="The output interval list file to write (if SCATTER_COUNT is 1) or the directory into which to write the scattered interval sub-directories (if SCATTER_COUNT > 1)", shortName="O", optional=true)
    public File OUTPUT;
    @Argument(doc="The amount to pad each end of the intervals by before other operations are undertaken. Negative numbers are allowed and indicate intervals should be shrunk. Resulting intervals < 0 bases long will be removed. Padding is applied to the interval lists <b> before </b> the ACTION is performed.", optional=true)
    public int PADDING = 0;
    @Argument(doc="If true, merge overlapping and adjacent intervals to create a list of unique intervals. Implies SORT=true")
    public boolean UNIQUE = false;
    @Argument(doc="If true, sort the resulting interval list by coordinate.")
    public boolean SORT = true;
    @Argument(doc="Action to take on inputs.")
    public Action ACTION = Action.CONCAT;
    @Argument(shortName="SI", doc="Second set of intervals for SUBTRACT and DIFFERENCE operations.", optional=true)
    public List<File> SECOND_INPUT;
    @Argument(doc="One or more lines of comment to add to the header of the output file.", optional=true)
    public List<String> COMMENT = null;
    @Argument(doc="The number of files into which to scatter the resulting list by locus; in some situations, fewer intervals may be emitted.  Note - if > 1, the resultant scattered intervals will be sorted and uniqued.  The sort will be inverted if the INVERT flag is set.")
    public int SCATTER_COUNT = 1;
    @Argument(doc="Whether to include filtered variants in the vcf when generating an interval list from vcf", optional=true)
    public boolean INCLUDE_FILTERED = false;
    @Argument(shortName="BRK", doc="If set to a positive value will create a new interval list with the original intervals broken up at integer multiples of this value.  Set to 0 to NOT break up intervals", optional=true)
    public int BREAK_BANDS_AT_MULTIPLES_OF = 0;
    @Argument(shortName="M", doc="Do not subdivide ")
    public IntervalListScatterer.Mode SUBDIVISION_MODE = IntervalListScatterer.Mode.INTERVAL_SUBDIVISION;
    @Argument(doc="Produce the inverse list", optional=true)
    public boolean INVERT = false;
    private static final Log LOG = Log.getInstance(IntervalListTools.class);

    public static void main(String[] args) {
        new IntervalListTools().instanceMainWithExit(args);
    }

    /*
     * WARNING - void declaration
     */
    @Override
    protected int doWork() {
        void var10_22;
        List finalIntervals;
        for (File f : this.INPUT) {
            IOUtil.assertFileIsReadable((File)f);
        }
        for (File f : this.SECOND_INPUT) {
            IOUtil.assertFileIsReadable((File)f);
        }
        if (this.OUTPUT != null) {
            if (this.SCATTER_COUNT == 1) {
                IOUtil.assertFileIsWritable((File)this.OUTPUT);
            } else {
                IOUtil.assertDirectoryIsWritable((File)this.OUTPUT);
            }
        }
        List<IntervalList> lists = this.openIntervalLists(this.INPUT);
        List<IntervalList> secondLists = this.openIntervalLists(this.SECOND_INPUT);
        if (this.UNIQUE && !this.SORT) {
            LOG.warn(new Object[]{"UNIQUE=true requires sorting but SORT=false was specified.  Results will be sorted!"});
        }
        IntervalList result = this.ACTION.act(lists, secondLists);
        if (this.SCATTER_COUNT > 1) {
            this.SORT = true;
            this.UNIQUE = true;
        }
        if (this.INVERT) {
            this.SORT = false;
            this.UNIQUE = true;
        }
        IntervalList possiblySortedResult = this.SORT ? result.sorted() : result;
        IntervalList possiblyInvertedResult = this.INVERT ? IntervalList.invert((IntervalList)possiblySortedResult) : possiblySortedResult;
        List list = finalIntervals = this.UNIQUE ? possiblyInvertedResult.uniqued().getIntervals() : possiblyInvertedResult.getIntervals();
        if (this.BREAK_BANDS_AT_MULTIPLES_OF > 0) {
            finalIntervals = IntervalList.breakIntervalsAtBandMultiples((List)finalIntervals, (int)this.BREAK_BANDS_AT_MULTIPLES_OF);
        }
        SAMFileHeader header = result.getHeader();
        HashSet<String> pgs = new HashSet<String>();
        for (SAMProgramRecord sAMProgramRecord : header.getProgramRecords()) {
            pgs.add(sAMProgramRecord.getId());
        }
        for (int i = 1; i < Integer.MAX_VALUE; ++i) {
            if (pgs.contains(String.valueOf(i))) continue;
            SAMProgramRecord sAMProgramRecord = new SAMProgramRecord(String.valueOf(i));
            sAMProgramRecord.setCommandLine(this.getCommandLine());
            sAMProgramRecord.setProgramName(this.getClass().getSimpleName());
            header.addProgramRecord(sAMProgramRecord);
            break;
        }
        if (this.COMMENT != null) {
            for (String string : this.COMMENT) {
                header.addComment(string);
            }
        }
        IntervalList output = new IntervalList(header);
        for (Interval i : finalIntervals) {
            output.add(i);
        }
        if (this.OUTPUT != null) {
            if (this.SCATTER_COUNT == 1) {
                output.write(this.OUTPUT);
                List<IntervalList> list2 = Arrays.asList(output);
            } else {
                List<IntervalList> scattered = this.writeScatterIntervals(output);
                LOG.info(new Object[]{String.format("Wrote %s scatter subdirectories to %s.", scattered.size(), this.OUTPUT)});
                if (scattered.size() != this.SCATTER_COUNT) {
                    LOG.warn(new Object[]{String.format("Requested scatter width of %s, but only emitted %s.  (This may be an expected consequence of running in %s mode.)", new Object[]{this.SCATTER_COUNT, scattered.size(), this.SUBDIVISION_MODE})});
                }
                List<IntervalList> list3 = scattered;
            }
        } else {
            List<IntervalList> list4 = Arrays.asList(output);
        }
        long totalUniqueBaseCount = 0L;
        long intervalCount = 0L;
        for (IntervalList finalInterval : var10_22) {
            totalUniqueBaseCount += finalInterval.getUniqueBaseCount();
            intervalCount += (long)finalInterval.size();
        }
        LOG.info(new Object[]{"Produced " + intervalCount + " intervals totalling " + totalUniqueBaseCount + " unique bases."});
        return 0;
    }

    private List<IntervalList> openIntervalLists(List<File> files) {
        ArrayList<IntervalList> lists = new ArrayList<IntervalList>();
        for (File f : files) {
            lists.add(TYPE.getIntervalList(f, this.INCLUDE_FILTERED).padded(this.PADDING));
        }
        return lists;
    }

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> errorMsgs = new ArrayList<String>();
        if (this.SCATTER_COUNT < 1) {
            errorMsgs.add("SCATTER_COUNT must be greater than 0.");
        }
        if (this.BREAK_BANDS_AT_MULTIPLES_OF < 0) {
            errorMsgs.add("BREAK_BANDS_AT_MULTIPLES_OF must be greater than or equal to 0.");
        }
        return errorMsgs.isEmpty() ? null : errorMsgs.toArray(new String[errorMsgs.size()]);
    }

    private List<IntervalList> writeScatterIntervals(IntervalList list) {
        IntervalListScatterer scatterer = new IntervalListScatterer(this.SUBDIVISION_MODE);
        List<IntervalList> scattered = scatterer.scatter(list, this.SCATTER_COUNT, this.UNIQUE);
        DecimalFormat fileNameFormatter = new DecimalFormat("0000");
        int fileIndex = 1;
        for (IntervalList intervals : scattered) {
            intervals.write(IntervalListTools.createDirectoryAndGetScatterFile(this.OUTPUT, scattered.size(), fileNameFormatter.format(fileIndex++)));
        }
        return scattered;
    }

    public static File getScatteredFileName(File scatterDirectory, long scatterTotal, String formattedIndex) {
        return new File(scatterDirectory.getAbsolutePath() + "/temp_" + formattedIndex + "_of_" + scatterTotal + "/scattered" + ".interval_list");
    }

    private static File createDirectoryAndGetScatterFile(File outputDirectory, long scatterCount, String formattedIndex) {
        IntervalListTools.createDirectoryOrFail(outputDirectory);
        File result = IntervalListTools.getScatteredFileName(outputDirectory, scatterCount, formattedIndex);
        IntervalListTools.createDirectoryOrFail(result.getParentFile());
        return result;
    }

    private static void createDirectoryOrFail(File directory) {
        if (!directory.exists() && !directory.mkdir()) {
            throw new PicardException("Unable to create directory: " + directory.getAbsolutePath());
        }
    }

    static enum TYPE {
        VCF(IOUtil.VCF_EXTENSIONS){

            @Override
            protected IntervalList getIntervalListInternal(File vcf, boolean includeFiltered) {
                return VCFFileReader.fromVcf((File)vcf, (boolean)includeFiltered);
            }
        }
        ,
        INTERVAL_LIST(new String[]{".interval_list"}){

            @Override
            protected IntervalList getIntervalListInternal(File intervalList, boolean includeFiltered) {
                return IntervalList.fromFile((File)intervalList);
            }
        };

        final Collection<String> applicableExtensions;

        private TYPE(String ... s) {
            this.applicableExtensions = CollectionUtil.makeSet((Object[])s);
        }

        private TYPE(Collection<String> extensions) {
            this.applicableExtensions = extensions;
        }

        protected abstract IntervalList getIntervalListInternal(File var1, boolean var2);

        static TYPE forFile(File intervalListExtractable) {
            for (TYPE type : TYPE.values()) {
                for (String s : type.applicableExtensions) {
                    if (!intervalListExtractable.getName().endsWith(s)) continue;
                    return type;
                }
            }
            throw new SAMException("Cannot figure out type of file " + intervalListExtractable.getAbsolutePath() + " from extension. Current implementation understands the following types: " + Arrays.toString((Object[])TYPE.values()));
        }

        public static IntervalList getIntervalList(File file, boolean includeFiltered) {
            return TYPE.forFile(file).getIntervalListInternal(file, includeFiltered);
        }

        public String toString() {
            return super.toString() + ": " + this.applicableExtensions.toString();
        }
    }

    public static enum Action implements CommandLineParser.ClpEnum
    {
        CONCAT("The concatenation of all the INPUTs, no sorting or merging of overlapping/abutting intervals implied. Will result in an unsorted list unless requested otherwise."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.concatenate(list);
            }
        }
        ,
        UNION("Like CONCATENATE but with UNIQUE and SORT implied, the result being the set-wise union of all INPUTS."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.union(list);
            }
        }
        ,
        INTERSECT("The sorted, uniqued set of all loci that are contained in all of the INPUTs."){

            @Override
            IntervalList act(List<IntervalList> list, List<IntervalList> unused) {
                if (!unused.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Second List found when action was %s. Ignoring second list.", this.name()));
                }
                return IntervalList.intersection(list);
            }
        }
        ,
        SUBTRACT("Subtracts SECOND_INPUT from INPUT. The resulting loci are there in INPUT that are not in SECOND_INPUT"){

            @Override
            IntervalList act(List<IntervalList> list1, List<IntervalList> list2) {
                return IntervalList.subtract(list1, list2);
            }
        }
        ,
        SYMDIFF("Find loci that are in INPUT or SECOND_INPUT but are not in both."){

            @Override
            IntervalList act(List<IntervalList> list1, List<IntervalList> list2) {
                return IntervalList.difference(list1, list2);
            }
        }
        ,
        OVERLAPS("Find only intervals in INPUT that have any overlap with SECOND_INPUT"){

            @Override
            IntervalList act(List<IntervalList> list1, List<IntervalList> list2) {
                return IntervalList.overlaps(list1, list2);
            }
        };

        String helpdoc;

        private Action(String helpdoc) {
            this.helpdoc = helpdoc;
        }

        public String getHelpDoc() {
            return this.helpdoc;
        }

        abstract IntervalList act(List<IntervalList> var1, List<IntervalList> var2);
    }
}

