/*
 * Decompiled with CFR 0.152.
 */
package picard.analysis;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SAMRecordIterator;
import htsjdk.samtools.SAMTag;
import htsjdk.samtools.SamPairUtil;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.analysis.JumpingLibraryMetrics;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.DiagnosticsAndQCProgramGroup;
import picard.sam.DuplicationMetrics;

@CommandLineProgramProperties(summary="Collect jumping library metrics. <p>This tool collects high-level metrics about the presence of outward-facing (jumping) and inward-facing (non-jumping) read pairs within a SAM or BAM file.For a brief primer on jumping libraries, see the GATK <a href='https://www.broadinstitute.org/gatk/guide/article?id=6326'>Dictionary</a></p>.<p>This program gets all data for computation from the first read in each pair in which the mapping quality (MQ) tag is set with the mate's mapping quality.  If the MQ tag is not set, then the program assumes that the mate's MQ is greater than or equal to MINIMUM_MAPPING_QUALITY (default value is 0).</p> <p>Note: Metrics labeled as percentages are actually expressed as fractions!</p><h4>Usage example:</h4><pre>java -jar picard.jar CollectJumpingLibraryMetrics \\<br />      I=input.bam  \\<br />      O=jumping_metrics.txt</pre>Please see the output metrics documentation on <a href='https://broadinstitute.github.io/picard/picard-metric-definitions.html#JumpingLibraryMetrics'>JumpingLibraryMetrics</a> for detailed explanations of the output metrics.<hr />", oneLineSummary="Collect jumping library metrics. ", programGroup=DiagnosticsAndQCProgramGroup.class)
@DocumentedFeature
public class CollectJumpingLibraryMetrics
extends CommandLineProgram {
    static final String USAGE_SUMMARY = "Collect jumping library metrics. ";
    static final String USAGE_DETAILS = "<p>This tool collects high-level metrics about the presence of outward-facing (jumping) and inward-facing (non-jumping) read pairs within a SAM or BAM file.For a brief primer on jumping libraries, see the GATK <a href='https://www.broadinstitute.org/gatk/guide/article?id=6326'>Dictionary</a></p>.<p>This program gets all data for computation from the first read in each pair in which the mapping quality (MQ) tag is set with the mate's mapping quality.  If the MQ tag is not set, then the program assumes that the mate's MQ is greater than or equal to MINIMUM_MAPPING_QUALITY (default value is 0).</p> <p>Note: Metrics labeled as percentages are actually expressed as fractions!</p><h4>Usage example:</h4><pre>java -jar picard.jar CollectJumpingLibraryMetrics \\<br />      I=input.bam  \\<br />      O=jumping_metrics.txt</pre>Please see the output metrics documentation on <a href='https://broadinstitute.github.io/picard/picard-metric-definitions.html#JumpingLibraryMetrics'>JumpingLibraryMetrics</a> for detailed explanations of the output metrics.<hr />";
    @Argument(shortName="I", doc="BAM file(s) of reads with duplicates marked")
    public List<File> INPUT = new ArrayList<File>();
    @Argument(shortName="O", doc="File to which metrics should be written")
    public File OUTPUT;
    @Argument(shortName="MQ", doc="Mapping quality minimum cutoff")
    public Integer MINIMUM_MAPPING_QUALITY = 0;
    @Argument(shortName="T", doc="When calculating mean and stdev stop when the bins in the tail of the distribution contain fewer than mode/TAIL_LIMIT items")
    public int TAIL_LIMIT = 10000;
    @Argument(doc="Jumps greater than or equal to the greater of this value or 2 times the mode of the outward-facing pairs are considered chimeras")
    public int CHIMERA_KB_MIN = 100000;
    private static final int SAMPLE_FOR_MODE = 50000;

    @Override
    protected int doWork() {
        for (File f : this.INPUT) {
            IOUtil.assertFileIsReadable((File)f);
        }
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        Histogram innieHistogram = new Histogram();
        Histogram outieHistogram = new Histogram();
        int fragments = 0;
        int innies = 0;
        int outies = 0;
        int innieDupes = 0;
        int outieDupes = 0;
        int crossChromPairs = 0;
        int superSized = 0;
        int tandemPairs = 0;
        double chimeraSizeMinimum = Math.max(this.getOutieMode(), (double)this.CHIMERA_KB_MIN);
        for (File f : this.INPUT) {
            SamReader reader = SamReaderFactory.makeDefault().open(f);
            if (reader.getFileHeader().getSortOrder() != SAMFileHeader.SortOrder.coordinate) {
                throw new PicardException("SAM file must " + f.getName() + " must be sorted in coordintate order");
            }
            for (SAMRecord sam : reader) {
                if (!sam.getFirstOfPairFlag()) continue;
                if (sam.getReadUnmappedFlag()) {
                    if (!sam.getMateUnmappedFlag()) {
                        ++fragments;
                        continue;
                    }
                    if (sam.getReferenceIndex() != -1) continue;
                    break;
                }
                if (sam.getMateUnmappedFlag()) {
                    ++fragments;
                    continue;
                }
                if (sam.getAttribute(SAMTag.MQ.name()) != null && sam.getIntegerAttribute(SAMTag.MQ.name()) < this.MINIMUM_MAPPING_QUALITY || sam.getMappingQuality() < this.MINIMUM_MAPPING_QUALITY) continue;
                int absInsertSize = Math.abs(sam.getInferredInsertSize());
                if ((double)absInsertSize > chimeraSizeMinimum) {
                    ++superSized;
                    continue;
                }
                if (sam.getMateNegativeStrandFlag() == sam.getReadNegativeStrandFlag()) {
                    ++tandemPairs;
                    continue;
                }
                if (!sam.getMateReferenceIndex().equals(sam.getReferenceIndex())) {
                    ++crossChromPairs;
                    continue;
                }
                SamPairUtil.PairOrientation pairOrientation = SamPairUtil.getPairOrientation((SAMRecord)sam);
                if (pairOrientation == SamPairUtil.PairOrientation.RF) {
                    outieHistogram.increment((Comparable)Integer.valueOf(absInsertSize));
                    ++outies;
                    if (!sam.getDuplicateReadFlag()) continue;
                    ++outieDupes;
                    continue;
                }
                if (pairOrientation == SamPairUtil.PairOrientation.FR) {
                    innieHistogram.increment((Comparable)Integer.valueOf(absInsertSize));
                    ++innies;
                    if (!sam.getDuplicateReadFlag()) continue;
                    ++innieDupes;
                    continue;
                }
                throw new IllegalStateException("This should never happen");
            }
            CloserUtil.close((Object)reader);
        }
        MetricsFile metricsFile = this.getMetricsFile();
        JumpingLibraryMetrics metrics = new JumpingLibraryMetrics();
        metrics.JUMP_PAIRS = outies;
        metrics.JUMP_DUPLICATE_PAIRS = outieDupes;
        metrics.JUMP_DUPLICATE_PCT = outies != 0 ? (double)outieDupes / (double)outies : 0.0;
        metrics.JUMP_LIBRARY_SIZE = outies > 0 && outieDupes > 0 ? DuplicationMetrics.estimateLibrarySize(outies, outies - outieDupes) : 0L;
        outieHistogram.trimByTailLimit(this.TAIL_LIMIT);
        metrics.JUMP_MEAN_INSERT_SIZE = outieHistogram.getMean();
        metrics.JUMP_STDEV_INSERT_SIZE = outieHistogram.getStandardDeviation();
        metrics.NONJUMP_PAIRS = innies;
        metrics.NONJUMP_DUPLICATE_PAIRS = innieDupes;
        metrics.NONJUMP_DUPLICATE_PCT = innies != 0 ? (double)innieDupes / (double)innies : 0.0;
        metrics.NONJUMP_LIBRARY_SIZE = innies > 0 && innieDupes > 0 ? DuplicationMetrics.estimateLibrarySize(innies, innies - innieDupes) : 0L;
        innieHistogram.trimByTailLimit(this.TAIL_LIMIT);
        metrics.NONJUMP_MEAN_INSERT_SIZE = innieHistogram.getMean();
        metrics.NONJUMP_STDEV_INSERT_SIZE = innieHistogram.getStandardDeviation();
        metrics.CHIMERIC_PAIRS = crossChromPairs + superSized + tandemPairs;
        metrics.FRAGMENTS = fragments;
        double totalPairs = (long)(outies + innies) + metrics.CHIMERIC_PAIRS;
        metrics.PCT_JUMPS = totalPairs != 0.0 ? (double)outies / totalPairs : 0.0;
        metrics.PCT_NONJUMPS = totalPairs != 0.0 ? (double)innies / totalPairs : 0.0;
        metrics.PCT_CHIMERAS = totalPairs != 0.0 ? (double)metrics.CHIMERIC_PAIRS / totalPairs : 0.0;
        metricsFile.addMetric((MetricBase)metrics);
        metricsFile.write(this.OUTPUT);
        return 0;
    }

    private double getOutieMode() {
        int samplePerFile = 50000 / this.INPUT.size();
        Histogram histo = new Histogram();
        for (File f : this.INPUT) {
            SamReader reader = SamReaderFactory.makeDefault().open(f);
            int sampled = 0;
            SAMRecordIterator it = reader.iterator();
            while (it.hasNext() && sampled < samplePerFile) {
                SAMRecord sam = (SAMRecord)it.next();
                if (!sam.getFirstOfPairFlag()) continue;
                if (sam.getReadUnmappedFlag() && sam.getReferenceIndex() == -1) break;
                if (sam.getReadUnmappedFlag() || sam.getMateUnmappedFlag() || sam.getAttribute(SAMTag.MQ.name()) != null && sam.getIntegerAttribute(SAMTag.MQ.name()) < this.MINIMUM_MAPPING_QUALITY || sam.getMappingQuality() < this.MINIMUM_MAPPING_QUALITY || sam.getMateNegativeStrandFlag() == sam.getReadNegativeStrandFlag() || !sam.getMateReferenceIndex().equals(sam.getReferenceIndex()) || SamPairUtil.getPairOrientation((SAMRecord)sam) != SamPairUtil.PairOrientation.RF) continue;
                histo.increment((Comparable)Integer.valueOf(Math.abs(sam.getInferredInsertSize())));
                ++sampled;
            }
            CloserUtil.close((Object)reader);
        }
        return histo.size() > 0 ? histo.getMode() : 0.0;
    }
}

