/*
 * Decompiled with CFR 0.152.
 */
package picard.sam;

import htsjdk.samtools.SAMFileHeader;
import htsjdk.samtools.SAMFileWriter;
import htsjdk.samtools.SAMFileWriterFactory;
import htsjdk.samtools.SAMProgramRecord;
import htsjdk.samtools.SAMRecord;
import htsjdk.samtools.SamReader;
import htsjdk.samtools.SamReaderFactory;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.CollectionUtil;
import htsjdk.samtools.util.Histogram;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Log;
import htsjdk.samtools.util.ProgressLogger;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.PicardException;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.ReadDataManipulationProgramGroup;
import picard.sam.util.PhysicalLocationInt;
import picard.sam.util.ReadNameParser;

@CommandLineProgramProperties(summary="<h3>Summary</h3>\nClass to downsample a SAM/BAM file based on the position of the read in a flowcell. As with DownsampleSam, all the reads with the same queryname are either kept or dropped as a unit.\n\n <h3>Details</h3>\nThe downsampling is _not_ random (and there is no random seed). It is deterministically determined by the position of each read within its tile. Specifically, it draws an ellipse that covers a FRACTION of the total tile's area and of all the edges of the tile. It uses this area to determine whether to keep or drop the record. Since reads with the same name have the same position (mates, secondary and supplemental alignments), the decision will be the same for all of them. The main concern of this downsampling method is that due to \"optical duplicates\" downsampling randomly can create a result that has a different optical duplicate rate, and therefore a different estimated library size (when running MarkDuplicates). This method keeps (physically) close read together, so that (except for reads near the boundary of the circle) optical duplicates are kept or dropped as a group. By default the program expects the read names to have 5 or 7 fields separated by colons (:) and it takes the last two to indicate the x and y coordinates of the reads within the tile whence it was sequenced. See DEFAULT_READ_NAME_REGEX for more detail. The program traverses the INPUT twice: first to find out the size of each of the tiles, and next to perform the downsampling. Downsampling invalidates the duplicate flag because duplicate reads before downsampling may not all remain duplicated after downsampling. Thus, the default setting also removes the duplicate information. \n\nExample\n\njava -jar picard.jar PositionBasedDownsampleSam \\\n      I=input.bam \\\n      O=downsampled.bam \\\n      FRACTION=0.1\n\nCaveats\n\nNote 1: This method is <b>technology and read-name dependent</b>. If the read-names do not have coordinate information embedded in them, or if your BAM contains reads from multiple technologies (flowcell versions, sequencing machines) this will not work properly. It has been designed to work with Illumina technology and reads-names. Consider modifying {@link #READ_NAME_REGEX} in other cases. \n\nNote 2: The code has been designed to simulate, as accurately as possible, sequencing less, <b>not</b> for getting an exact downsampled fraction (Use {@link DownsampleSam} for that.) In particular, since the reads may be distributed non-evenly within the lanes/tiles, the resulting downsampling percentage will not be accurately determined by the input argument FRACTION. \n\nNote 3:Consider running {@link MarkDuplicates} after downsampling in order to \"expose\" the duplicates whose representative has been downsampled away.\n\nNote 4:The downsampling assumes a uniform distribution of reads in the flowcell. Input already downsampled with PositionBasedDownsampleSam violates this assumption. To guard against such input, PositionBasedDownsampleSam always places a PG record in the header of its output, and aborts whenever it finds such a PG record in its input.", oneLineSummary="Downsample a SAM or BAM file to retain a subset of the reads based on the reads location in each tile in the flowcell.", programGroup=ReadDataManipulationProgramGroup.class)
@DocumentedFeature
public class PositionBasedDownsampleSam
extends CommandLineProgram {
    @Argument(shortName="I", doc="The input SAM or BAM file to downsample.")
    public File INPUT;
    @Argument(shortName="O", doc="The output, downsampled, SAM or BAM file.")
    public File OUTPUT;
    @Argument(shortName="F", doc="The (approximate) fraction of reads to be kept, between 0 and 1.")
    public Double FRACTION = null;
    @Argument(doc="Determines whether the duplicate tag should be reset since the downsampling requires re-marking duplicates.")
    public boolean REMOVE_DUPLICATE_INFORMATION = true;
    @Argument(doc="Use these regular expressions to parse read names in the input SAM file. Read names are parsed to extract three variables: tile/region, x coordinate and y coordinate. The x and y coordinates are used to determine the downsample decision. Set this option to null to disable optical duplicate detection, e.g. for RNA-seq The regular expression should contain three capture groups for the three variables, in order. It must match the entire read name. Note that if the default regex is specified, a regex match is not actually done, but instead the read name is split on colons (:). For 5 element names, the 3rd, 4th and 5th elements are assumed to be tile, x and y values. For 7 element names (CASAVA 1.8), the 5th, 6th, and 7th elements are assumed to be tile, x and y values.")
    public String READ_NAME_REGEX = ReadNameParser.DEFAULT_READ_NAME_REGEX;
    @Argument(doc="Stop after processing N reads, mainly for debugging.", optional=true)
    public Long STOP_AFTER = null;
    @Argument(doc="Allow downsampling again despite this being a bad idea with possibly unexpected results.", optional=true)
    public boolean ALLOW_MULTIPLE_DOWNSAMPLING_DESPITE_WARNINGS = false;
    private final Log log = Log.getInstance(PositionBasedDownsampleSam.class);
    private ReadNameParser readNameParser;
    private long total = 0L;
    private long kept = 0L;
    public static String PG_PROGRAM_NAME = "PositionBasedDownsampleSam";
    private static final double ACCEPTABLE_FUDGE_FACTOR = 0.2;
    CollectionUtil.DefaultingMap.Factory<Coord, Short> defaultingMapFactory = aShort -> new Coord();
    private final Map<Short, Coord> tileCoord = new CollectionUtil.DefaultingMap(this.defaultingMapFactory, true);
    final Map<Short, Histogram<Short>> xPositions = new HashMap<Short, Histogram<Short>>();
    final Map<Short, Histogram<Short>> yPositions = new HashMap<Short, Histogram<Short>>();

    @Override
    protected String[] customCommandLineValidation() {
        ArrayList<String> errors = new ArrayList<String>();
        if (this.FRACTION < 0.0 || this.FRACTION > 1.0) {
            errors.add("FRACTION must be a value between 0 and 1, found: " + this.FRACTION);
        }
        if (!errors.isEmpty()) {
            return errors.toArray(new String[errors.size()]);
        }
        return super.customCommandLineValidation();
    }

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable((File)this.INPUT);
        IOUtil.assertFileIsWritable((File)this.OUTPUT);
        this.log.info(new Object[]{"Checking to see if input file has been downsampled with this program before."});
        this.checkProgramRecords();
        this.readNameParser = new ReadNameParser(this.READ_NAME_REGEX);
        this.log.info(new Object[]{"Starting first pass. Examining read distribution in tiles."});
        this.fillTileMinMaxCoord();
        this.log.info(new Object[]{"First pass done."});
        this.log.info(new Object[]{"Starting second pass. Outputting reads."});
        this.outputSamRecords();
        this.log.info(new Object[]{"Second pass done."});
        double finalP = (double)this.kept / (double)this.total;
        if (Math.abs(finalP - this.FRACTION) / (Math.min(finalP, this.FRACTION) + 1.0E-10) > 0.2) {
            this.log.warn(new Object[]{String.format("You've requested FRACTION=%g, the resulting downsampling resulted in a rate of %f.", this.FRACTION, finalP)});
        }
        this.log.info(new Object[]{String.format("Finished! Kept %d out of %d reads (P=%g).", this.kept, this.total, finalP)});
        return 0;
    }

    private void outputSamRecords() {
        ProgressLogger progress = new ProgressLogger(this.log, 10000000);
        SamReader in = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        SAMFileHeader header = in.getFileHeader().clone();
        SAMFileHeader.PgIdGenerator pgIdGenerator = new SAMFileHeader.PgIdGenerator(header);
        SAMProgramRecord programRecord = new SAMProgramRecord(pgIdGenerator.getNonCollidingId(PG_PROGRAM_NAME));
        programRecord.setProgramName(PG_PROGRAM_NAME);
        programRecord.setCommandLine(this.getCommandLine());
        programRecord.setProgramVersion(this.getVersion());
        header.addProgramRecord(programRecord);
        SAMFileWriter out = new SAMFileWriterFactory().makeSAMOrBAMWriter(header, true, this.OUTPUT);
        CircleSelector selector = new CircleSelector(this.FRACTION);
        for (SAMRecord rec : in) {
            boolean keepRecord;
            if (this.STOP_AFTER != null && this.total >= this.STOP_AFTER) break;
            ++this.total;
            PhysicalLocationInt pos = this.getSamRecordLocation(rec);
            if (!this.xPositions.containsKey(pos.getTile())) {
                this.xPositions.put(pos.getTile(), (Histogram<Short>)new Histogram(pos.getTile() + "-xpos", "count"));
            }
            if (!this.yPositions.containsKey(pos.getTile())) {
                this.yPositions.put(pos.getTile(), (Histogram<Short>)new Histogram(pos.getTile() + "-ypos", "count"));
            }
            if (keepRecord = selector.select(pos, this.tileCoord.get(pos.getTile()))) {
                if (this.REMOVE_DUPLICATE_INFORMATION) {
                    rec.setDuplicateReadFlag(false);
                }
                out.addAlignment(rec);
                ++this.kept;
            }
            progress.record(rec);
        }
        out.close();
        CloserUtil.close((Object)in);
    }

    private void checkProgramRecords() {
        SamReader in = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        for (SAMProgramRecord pg : in.getFileHeader().getProgramRecords()) {
            if (pg.getProgramName() == null || !pg.getProgramName().equals(PG_PROGRAM_NAME)) continue;
            String outText = "Found previous Program Record that indicates that this BAM has been downsampled already with this program. Operation not supported! Previous PG: " + pg.toString();
            if (this.ALLOW_MULTIPLE_DOWNSAMPLING_DESPITE_WARNINGS) {
                this.log.warn(new Object[]{outText});
                continue;
            }
            this.log.error(new Object[]{outText});
            throw new PicardException(outText);
        }
        CloserUtil.close((Object)in);
    }

    private void fillTileMinMaxCoord() {
        SamReader in = SamReaderFactory.makeDefault().referenceSequence(this.REFERENCE_SEQUENCE).open(this.INPUT);
        ProgressLogger progress = new ProgressLogger(this.log, 10000000, "Read");
        int total = 0;
        for (SAMRecord rec : in) {
            if (this.STOP_AFTER != null && (long)total >= this.STOP_AFTER) break;
            ++total;
            progress.record(rec);
            PhysicalLocationInt location = this.getSamRecordLocation(rec);
            Coord Pos2 = this.tileCoord.get(location.getTile());
            Pos2.maxX = Math.max(Pos2.maxX, location.getX());
            Pos2.minX = Math.min(Pos2.minX, location.getX());
            Pos2.maxY = Math.max(Pos2.maxY, location.getY());
            Pos2.minY = Math.min(Pos2.minY, location.getY());
            ++Pos2.count;
        }
        for (Coord coord : this.tileCoord.values()) {
            int diffX = coord.maxX - coord.minX;
            int diffY = coord.maxY - coord.minY;
            coord.maxX += diffX / coord.count;
            coord.minX -= diffX / coord.count;
            coord.maxY += diffY / coord.count;
            coord.minY -= diffY / coord.count;
        }
        CloserUtil.close((Object)in);
    }

    private PhysicalLocationInt getSamRecordLocation(SAMRecord rec) {
        PhysicalLocationInt pos = new PhysicalLocationInt();
        this.readNameParser.addLocationInformation(rec.getReadName(), pos);
        return pos;
    }

    private class Coord {
        public int minX = 0;
        public int minY = 0;
        public int maxX = 0;
        public int maxY = 0;
        public int count = 0;
    }

    private class CircleSelector {
        private final double radiusSquared;
        private final double offset;
        private final boolean positiveSelection;

        CircleSelector(double fraction) {
            double p;
            if (fraction > 0.5) {
                p = 1.0 - fraction;
                this.positiveSelection = false;
            } else {
                p = fraction;
                this.positiveSelection = true;
            }
            this.radiusSquared = p / Math.PI;
            if (p < 0.0) {
                throw new PicardException("This shouldn't happen...");
            }
            this.offset = Math.sqrt(this.radiusSquared - p * p / 4.0);
        }

        private double roundedPart(double x) {
            return x - (double)Math.round(x);
        }

        private boolean select(PhysicalLocationInt coord, Coord tileCoord) {
            double distanceSquared = Math.pow(this.roundedPart((double)(coord.getX() - tileCoord.minX) / (double)(tileCoord.maxX - tileCoord.minX) - this.offset), 2.0) + Math.pow(this.roundedPart((double)(coord.getY() - tileCoord.minY) / (double)(tileCoord.maxY - tileCoord.minY) - this.offset), 2.0);
            return distanceSquared > this.radiusSquared ^ this.positiveSelection;
        }
    }
}

