/*
 * Decompiled with CFR 0.152.
 */
package picard.arrays;

import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.metrics.MetricBase;
import htsjdk.samtools.metrics.MetricsFile;
import htsjdk.samtools.util.CloserUtil;
import htsjdk.samtools.util.IOUtil;
import htsjdk.samtools.util.Iso8601Date;
import htsjdk.samtools.util.Log;
import htsjdk.variant.utils.SAMSequenceDictionaryExtractor;
import htsjdk.variant.vcf.VCFFileReader;
import htsjdk.variant.vcf.VCFHeader;
import htsjdk.variant.vcf.VCFHeaderLine;
import java.io.File;
import java.nio.file.Path;
import java.util.Collection;
import java.util.EnumMap;
import org.apache.commons.lang.StringUtils;
import org.broadinstitute.barclay.argparser.Argument;
import org.broadinstitute.barclay.argparser.CommandLineProgramProperties;
import org.broadinstitute.barclay.help.DocumentedFeature;
import picard.analysis.MergeableMetricBase;
import picard.arrays.ArraysCallingMetricAccumulator;
import picard.arrays.illumina.ArraysControlInfo;
import picard.cmdline.CommandLineProgram;
import picard.cmdline.programgroups.DiagnosticsAndQCProgramGroup;
import picard.pedigree.Sex;
import picard.util.DbSnpBitSetUtil;
import picard.vcf.processor.VariantProcessor;

@CommandLineProgramProperties(summary="CollectArraysVariantCallingMetrics takes a Genotyping Arrays VCF file (as generated by GtcToVcf) and calculates summary and per-sample metrics. <h4>Usage example:</h4><pre>java -jar picard.jar CollectArraysVariantCallingMetrics \\<br />      INPUT=genotyping_arrays.vcf \\<br />      OUTPUT=outputBaseName</pre>", oneLineSummary="Collects summary and per-sample from the provided arrays VCF file", programGroup=DiagnosticsAndQCProgramGroup.class)
@DocumentedFeature
public class CollectArraysVariantCallingMetrics
extends CommandLineProgram {
    static final String USAGE_DETAILS = "CollectArraysVariantCallingMetrics takes a Genotyping Arrays VCF file (as generated by GtcToVcf) and calculates summary and per-sample metrics. <h4>Usage example:</h4><pre>java -jar picard.jar CollectArraysVariantCallingMetrics \\<br />      INPUT=genotyping_arrays.vcf \\<br />      OUTPUT=outputBaseName</pre>";
    @Argument(shortName="I", doc="Input vcf file for analysis")
    public File INPUT;
    @Argument(shortName="O", doc="Path (except for the file extension) of output metrics files to write.")
    public File OUTPUT;
    @Argument(doc="The Call Rate Threshold for an autocall pass (if the observed call rate is > this value, the sample is considered to be passing)", optional=true)
    public static Double CALL_RATE_PF_THRESHOLD = 0.98;
    @Argument(doc="Reference dbSNP file in dbSNP or VCF format.")
    public File DBSNP;
    @Argument(shortName="SD", optional=true, doc="If present, speeds loading of dbSNP file, will look for dictionary in vcf if not present here.")
    public File SEQUENCE_DICTIONARY = null;
    @Argument(doc="Split this task over multiple threads.  If NUM_PROCESSORS = 0, number of cores is automatically set to the number of cores available on the machine. If NUM_PROCESSORS < 0 then the number of cores used will be the number available on the machine less NUM_PROCESSORS.")
    public int NUM_PROCESSORS = 0;
    private final Log log = Log.getInstance(CollectArraysVariantCallingMetrics.class);

    @Override
    protected int doWork() {
        IOUtil.assertFileIsReadable((File)this.INPUT);
        IOUtil.assertFileIsReadable((File)this.DBSNP);
        if (this.SEQUENCE_DICTIONARY != null) {
            IOUtil.assertFileIsReadable((File)this.SEQUENCE_DICTIONARY);
        }
        VCFFileReader variantReader = new VCFFileReader(this.INPUT, true);
        VCFHeader vcfHeader = variantReader.getFileHeader();
        CloserUtil.close((Object)variantReader);
        SAMSequenceDictionary sequenceDictionary = SAMSequenceDictionaryExtractor.extractDictionary((Path)(this.SEQUENCE_DICTIONARY == null ? this.INPUT.toPath() : this.SEQUENCE_DICTIONARY.toPath()));
        this.log.info(new Object[]{"Loading dbSNP file."});
        DbSnpBitSetUtil.DbSnpBitSets dbsnp = DbSnpBitSetUtil.createSnpAndIndelBitSets(this.DBSNP, sequenceDictionary);
        this.log.info(new Object[]{"Starting iteration of variants."});
        ArraysControlInfo[] controlInfos = new ArraysControlInfo[ArraysControlInfo.CONTROL_INFO.length];
        for (int i = 0; i < ArraysControlInfo.CONTROL_INFO.length; ++i) {
            controlInfos[i] = this.parseControlHeaderString(this.getValueFromVcfOtherHeaderLine(vcfHeader, ArraysControlInfo.CONTROL_INFO[i].getControl()));
        }
        MetricsFile controlSummary = this.getMetricsFile();
        for (ArraysControlInfo info : controlInfos) {
            controlSummary.addMetric((MetricBase)new ArraysControlCodesSummaryMetrics(info.getControl(), info.getCategory(), info.getRed(), info.getGreen()));
        }
        VariantProcessor.Builder<ArraysCallingMetricAccumulator, ArraysCallingMetricAccumulator.Result> builder = VariantProcessor.Builder.generatingAccumulatorsBy(() -> {
            ArraysCallingMetricAccumulator accumulator = new ArraysCallingMetricAccumulator(dbsnp);
            accumulator.setup(vcfHeader);
            return accumulator;
        }).combiningResultsBy(ArraysCallingMetricAccumulator.Result::merge).withInput(this.INPUT).multithreadingBy(this.NUM_PROCESSORS);
        ArraysCallingMetricAccumulator.Result result = builder.build().process();
        MetricsFile detail = this.getMetricsFile();
        MetricsFile summary = this.getMetricsFile();
        summary.addMetric((MetricBase)result.summary);
        result.details.forEach(arg_0 -> detail.addMetric(arg_0));
        String outputPrefix = this.OUTPUT.getAbsolutePath() + ".";
        detail.write(new File(outputPrefix + ArraysVariantCallingDetailMetrics.getFileExtension()));
        summary.write(new File(outputPrefix + ArraysVariantCallingSummaryMetrics.getFileExtension()));
        controlSummary.write(new File(outputPrefix + ArraysControlCodesSummaryMetrics.getFileExtension()));
        return 0;
    }

    private ArraysControlInfo parseControlHeaderString(String controlString) {
        String[] tokens = controlString.split("\\|");
        return new ArraysControlInfo(tokens[0], tokens[1], Integer.valueOf(tokens[2]), Integer.valueOf(tokens[3]));
    }

    private String getValueFromVcfOtherHeaderLine(VCFHeader vcfHeader, String keyName) {
        VCFHeaderLine otherHeaderLine = vcfHeader.getOtherHeaderLine(keyName);
        if (otherHeaderLine != null) {
            return otherHeaderLine.getValue();
        }
        throw new IllegalArgumentException("Input VCF file is missing header line of type '" + keyName + "'");
    }

    @Override
    protected String[] customCommandLineValidation() {
        if (this.NUM_PROCESSORS == 0) {
            this.NUM_PROCESSORS = Runtime.getRuntime().availableProcessors();
        } else if (this.NUM_PROCESSORS < 0) {
            this.NUM_PROCESSORS = Runtime.getRuntime().availableProcessors() + this.NUM_PROCESSORS;
        }
        if (this.NUM_PROCESSORS <= 0) {
            this.NUM_PROCESSORS = 1;
        }
        if (CALL_RATE_PF_THRESHOLD <= 0.0 || CALL_RATE_PF_THRESHOLD > 1.0) {
            return new String[]{"The parameter CALL_RATE_PF_THRESHOLD must be > 0 and <= 1.0"};
        }
        return super.customCommandLineValidation();
    }

    public static boolean getSexConcordance(String reportedSexString, String fingerprintSexString, String autocallSexString) {
        Sex reportedSex = Sex.fromString(reportedSexString);
        Sex fingerprintSex = Sex.fromString(fingerprintSexString);
        Sex autocallSex = Sex.fromString(autocallSexString);
        EnumMap<Sex, Integer> sexMatchCount = new EnumMap<Sex, Integer>(Sex.class);
        for (Sex sex : Sex.values()) {
            sexMatchCount.put(sex, 0);
        }
        sexMatchCount.put(reportedSex, (Integer)sexMatchCount.get((Object)reportedSex) + 1);
        sexMatchCount.put(fingerprintSex, (Integer)sexMatchCount.get((Object)fingerprintSex) + 1);
        sexMatchCount.put(autocallSex, (Integer)sexMatchCount.get((Object)autocallSex) + 1);
        if ((Integer)sexMatchCount.get((Object)Sex.Unknown) + (Integer)sexMatchCount.get((Object)Sex.NotReported) == 3) {
            return false;
        }
        return (Integer)sexMatchCount.get((Object)Sex.Female) > 1 && (Integer)sexMatchCount.get((Object)Sex.Male) == 0 || (Integer)sexMatchCount.get((Object)Sex.Male) > 1 && (Integer)sexMatchCount.get((Object)Sex.Female) == 0;
    }

    @DocumentedFeature(groupName="Metrics", summary="Metrics")
    public static class ArraysVariantCallingDetailMetrics
    extends ArraysVariantCallingSummaryMetrics {
        @MergeableMetricBase.MergeByAssertEquals
        public String CHIP_WELL_BARCODE;
        @MergeableMetricBase.MergeByAssertEquals
        public String SAMPLE_ALIAS;
        @MergeableMetricBase.MergeByAssertEquals
        public Integer ANALYSIS_VERSION;
        @MergeableMetricBase.MergeByAssertEquals
        public String CHIP_TYPE;
        @MergeableMetricBase.NoMergingIsDerived
        public Boolean AUTOCALL_PF;
        @MergeableMetricBase.MergeByAssertEquals
        public Iso8601Date AUTOCALL_DATE;
        @MergeableMetricBase.MergeByAssertEquals
        public Iso8601Date IMAGING_DATE;
        @MergeableMetricBase.NoMergingIsDerived
        public Boolean IS_ZCALLED;
        @MergeableMetricBase.MergeByAssertEquals
        public Double GTC_CALL_RATE;
        @MergeableMetricBase.MergeByAssertEquals
        public String AUTOCALL_GENDER;
        @MergeableMetricBase.MergeByAssertEquals
        public String FP_GENDER;
        @MergeableMetricBase.MergeByAssertEquals
        public String REPORTED_GENDER;
        @MergeableMetricBase.MergeByAssertEquals
        public Boolean GENDER_CONCORDANCE_PF;
        @MergeableMetricBase.NoMergingIsDerived
        public double HET_PCT;
        @MergeableMetricBase.MergeByAssertEquals
        public String CLUSTER_FILE_NAME;
        @MergeableMetricBase.MergeByAssertEquals
        public Integer P95_GREEN;
        @MergeableMetricBase.MergeByAssertEquals
        public Integer P95_RED;
        @MergeableMetricBase.MergeByAssertEquals
        public String AUTOCALL_VERSION;
        @MergeableMetricBase.MergeByAssertEquals
        public String ZCALL_VERSION;
        @MergeableMetricBase.MergeByAssertEquals
        public String EXTENDED_MANIFEST_VERSION;
        @MergeableMetricBase.NoMergingIsDerived
        public double HET_HOMVAR_RATIO;
        @MergeableMetricBase.MergeByAssertEquals
        public String SCANNER_NAME;
        @MergeableMetricBase.MergeByAssertEquals
        public String PIPELINE_VERSION;
        @MergeableMetricBase.MergeByAdding
        long numHets;
        @MergeableMetricBase.MergeByAdding
        long numHomVar;
        @MergeableMetricBase.MergeByAssertEquals
        String zcallThresholdsFile;

        @Override
        public void calculateDerivedFields() {
            super.calculateDerivedFields();
            this.HET_HOMVAR_RATIO = (double)this.numHets / (double)this.numHomVar;
            this.HET_PCT = (double)this.numHets / (double)this.NUM_CALLS;
            this.GENDER_CONCORDANCE_PF = CollectArraysVariantCallingMetrics.getSexConcordance(this.REPORTED_GENDER, this.FP_GENDER, this.AUTOCALL_GENDER);
            this.IS_ZCALLED = !StringUtils.isEmpty((String)this.zcallThresholdsFile);
            this.AUTOCALL_PF = (double)this.AUTOCALL_CALL_RATE > CALL_RATE_PF_THRESHOLD;
        }

        public static String getFileExtension() {
            return "arrays_variant_calling_detail_metrics";
        }
    }

    @DocumentedFeature(groupName="Metrics", summary="Metrics")
    public static class ArraysControlCodesSummaryMetrics
    extends MetricBase {
        public String CONTROL;
        public String CATEGORY;
        public int RED;
        public int GREEN;

        public ArraysControlCodesSummaryMetrics() {
        }

        ArraysControlCodesSummaryMetrics(String control, String category, int red, int green) {
            this.CONTROL = control;
            this.CATEGORY = category;
            this.RED = red;
            this.GREEN = green;
        }

        public static String getFileExtension() {
            return "arrays_control_code_summary_metrics";
        }
    }

    @DocumentedFeature(groupName="Metrics", summary="Metrics")
    public static class ArraysVariantCallingSummaryMetrics
    extends MergeableMetricBase {
        @MergeableMetricBase.MergeByAdding
        public long NUM_ASSAYS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_NON_FILTERED_ASSAYS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_FILTERED_ASSAYS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_ZEROED_OUT_ASSAYS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_SNPS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_INDELS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_CALLS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_AUTOCALL_CALLS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_NO_CALLS;
        @MergeableMetricBase.MergeByAdding
        public long NUM_IN_DB_SNP;
        @MergeableMetricBase.MergeByAdding
        public long NOVEL_SNPS;
        @MergeableMetricBase.NoMergingIsDerived
        public float PCT_DBSNP;
        @MergeableMetricBase.NoMergingIsDerived
        public float CALL_RATE;
        @MergeableMetricBase.NoMergingIsDerived
        public float AUTOCALL_CALL_RATE;
        @MergeableMetricBase.MergeByAdding
        public long NUM_SINGLETONS;

        public static String getFileExtension() {
            return "arrays_variant_calling_summary_metrics";
        }

        @Override
        public void calculateDerivedFields() {
            this.PCT_DBSNP = (float)this.NUM_IN_DB_SNP / (float)this.NUM_SNPS;
            this.NOVEL_SNPS = this.NUM_SNPS - this.NUM_IN_DB_SNP;
            this.CALL_RATE = (float)this.NUM_CALLS / (float)this.NUM_NON_FILTERED_ASSAYS;
            this.AUTOCALL_CALL_RATE = (float)this.NUM_AUTOCALL_CALLS / (float)this.NUM_NON_FILTERED_ASSAYS;
        }

        public static <T extends ArraysVariantCallingSummaryMetrics> void foldInto(ArraysVariantCallingSummaryMetrics target, Collection<ArraysVariantCallingDetailMetrics> metrics) {
            metrics.forEach(target::merge);
        }
    }
}

